<?php

namespace Drupal\component_fields\Form;

use Drupal\component_fields\CompileService;
use Drupal\component_fields\ComponentFieldsCompilerPluginManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides a form for overriding field compilation settings.
 */
class OverrideFieldsForm extends ComponentFieldsBaseConfigForm {

  /**
   * The enabled bundles for component fields.
   *
   * @var array
   */
  protected array $enabledBundles;

  /**
   * The saved fields configuration.
   *
   * @var array
   */
  protected array $savedFieldsConfig = [];

  /**
   * The available string_long fields for overrides.
   *
   * @var array
   */
  protected array $availableStringLongFields;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'component_fields_override_fields_config_form';
  }

  /**
   * Constructs the OverrideFieldsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\component_fields\CompileService $compile_service
   *   The compile service.
   * @param \Drupal\component_fields\ComponentFieldsCompilerPluginManager $compiler_manager
   *   The component fields compiler plugin manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle info service.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    CompileService $compile_service,
    ComponentFieldsCompilerPluginManager $compiler_manager,
    EntityTypeManagerInterface $entity_type_manager,
    EntityTypeBundleInfoInterface $entity_type_bundle_info,
    EntityFieldManagerInterface $entity_field_manager,
  ) {
    parent::__construct($config_factory, $compile_service, $compiler_manager, $entity_type_manager, $entity_type_bundle_info, $entity_field_manager);
    $config = $this->config(ComponentFieldsBaseConfigForm::SETTINGS);
    $this->enabledBundles = $config->get('enabled_bundles') ?? [];
    $this->savedFieldsConfig = $config->get('fields_config') ?? [];
  }

  /**
   * Builds the override fields form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    if (empty($this->enabledBundles)) {
      $form['header'] = [
        '#markup' => $this->t('No bundles enabled.'),
      ];
      return $form;
    }

    $overrides = $this->config(ComponentFieldsBaseConfigForm::SETTINGS)
      ->get('overrides') ?? [];
    $form['#tree'] = TRUE;

    $form['header'] = [
      '#markup' => $this->t("If you want to be able to override the field compilation per entity, select a field to store the overrides. The bundle must have at least one string_long field available, that hasn\'t been used in any component field configuration. Only the bundles that have such a field, and at least one component field configuration, are listed."),
    ];

    $this->getStringLongFields();

    foreach ($this->enabledBundles as $entity_type_id => $bundles) {
      $entity_type_def = $this->entityTypeManager->getDefinition($entity_type_id);
      $entity_type_label = $entity_type_def->getLabel();
      $form[$entity_type_id] = [
        '#type' => 'details',
        '#title' => "$entity_type_label [$entity_type_id]",
        '#open' => TRUE,
      ];

      foreach ($bundles as $bundle_id) {
        if (empty($this->availableStringLongFields[$entity_type_id]) || empty($this->availableStringLongFields[$entity_type_id][$bundle_id])) {
          continue;
        }

        $bundle_def = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id)[$bundle_id];
        $bundle_label = $bundle_def['label'];

        $disabled = empty($this->savedFieldsConfig[$entity_type_id]) || empty($this->savedFieldsConfig[$entity_type_id][$bundle_id]);
        $available_overreide_fields = $this->availableStringLongFields[$entity_type_id][$bundle_id];
        $form[$entity_type_id][$bundle_id] = [
          '#type' => 'select',
          '#title' => "$bundle_label [$bundle_id]",
          '#options' => ['' => '-'] + $available_overreide_fields,
          '#default_value' => (!empty($overrides[$entity_type_id]) && !empty($overrides[$entity_type_id][$bundle_id])) ? $overrides[$entity_type_id][$bundle_id] : '',
          '#disabled' => $disabled,
        ];
        if ($disabled) {
          $form[$entity_type_id][$bundle_id]['#description'] = $this->t('This bundle has no fields configured, so no overrides can be set.');
        }
      }
    }

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Submits the override fields form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $override_fields = [];
    $values = $form_state->getValues();
    foreach ($this->enabledBundles as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle_id) {
        if (empty($values[$entity_type_id][$bundle_id])) {
          continue;
        }
        $field = $values[$entity_type_id][$bundle_id];
        $override_fields[$entity_type_id][$bundle_id] = $field;
      }
    }

    $this->config(ComponentFieldsBaseConfigForm::SETTINGS)
      ->set('overrides', $override_fields)
      ->save();

    $this->updateWidgets($override_fields);

    parent::submitForm($form, $form_state);
  }

  /**
   * Updates the widgets for the specified override fields.
   *
   * @param array $override_fields
   *   The override fields to update.
   */
  private function updateWidgets(array $override_fields) {
    foreach ($override_fields as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle_id => $field_name) {
        $form_displays = $this->entityTypeManager->getStorage('entity_form_display')
          ->loadByProperties([
            'targetEntityType' => $entity_type_id,
            'bundle' => $bundle_id,
          ]);
        foreach ($form_displays as $form_display_id => $form_display) {
          $mode = $form_display->getMode();
          $current_widget = $form_display->getComponent($field_name);
          if (empty($current_widget)) {
            continue;
          }
          $current_widget_type = $current_widget['type'];
          if ($current_widget_type === 'component_fields_override_widget') {
            continue;
          }
          $form_display->setComponent($field_name, [
            'type' => 'component_fields_override_widget',
          ]);
          $form_display->save();
        }
      }
    }
  }

  /**
   * Retrieves string_long fields for enabled bundles.
   */
  private function getStringLongFields(): void {
    foreach ($this->enabledBundles as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle_id) {
        $this->availableStringLongFields[$entity_type_id][$bundle_id] = $this->compileService->getStringLongFields($entity_type_id, $bundle_id);
        if (empty($this->availableStringLongFields[$entity_type_id][$bundle_id])) {
          continue;
        }
        $bundle_saved_config = $this->savedFieldsConfig[$entity_type_id][$bundle_id] ?? [];
        foreach ($bundle_saved_config as $triad) {
          $final_field = $triad["final"];
          unset($this->availableStringLongFields[$entity_type_id][$bundle_id][$final_field]);
          $component_1_field = $triad["component_1"];
          unset($this->availableStringLongFields[$entity_type_id][$bundle_id][$component_1_field]);
          $component_2_field = $triad["component_2"];
          unset($this->availableStringLongFields[$entity_type_id][$bundle_id][$component_2_field]);
        }
      }
    }
  }

}
