<?php

namespace Drupal\component_fields\Plugin\ComponentFieldsCompiler;

use Drupal\component_fields\ComponentFieldsCompilerBase;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;

/**
 * Merges the values from both components.
 *
 * This compiler takes values from 2 components, merges them into single array,
 * ensuring that duplicate values are removed.
 *
 * @ComponentFieldsCompiler(
 *   id = "merge",
 *   label = @Translation("Merge"),
 *   description = @Translation("Merge the values from both components."),
 *   multivalue_only = TRUE
 * )
 */
final class Merge extends ComponentFieldsCompilerBase {

  /**
   * Compiles the values by merging them into a single array.
   *
   * @param mixed $component_1_value
   *   The value from Component 1, expected to be an array.
   * @param mixed $component_2_value
   *   The value from Component 2, expected to be an array.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition associated with the components.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity being processed.
   *
   * @return array
   *   An array containing the merged values, with duplicates removed.
   */
  public function compile(mixed $component_1_value, mixed $component_2_value, FieldDefinitionInterface $field_definition, ContentEntityInterface $entity): array {
    if (count($component_1_value) && count($component_2_value)) {
      $merged = array_merge($component_1_value, $component_2_value);
      return $this->dedupe($merged);
    }
    elseif (count($component_1_value) && !count($component_2_value)) {
      return $component_1_value;
    }
    elseif (!count($component_1_value) && count($component_2_value)) {
      return $component_2_value;
    }
    else {
      return [];
    }
  }

  /**
   * Removes duplicate values from the merged array.
   *
   * @param array $merged
   *   The array containing merged values.
   *
   * @return array
   *   An array with duplicates removed.
   */
  private function dedupe(array $merged): array {
    $temp = [];
    $deduped = [];
    foreach ($merged as $i => $value_construct) {
      foreach ($value_construct as $key => $value) {
        if (array_key_exists($value, $temp)) {
          continue;
        }
        $temp[$value] = $value;
        $deduped[$i][$key] = $value;
      }
    }
    return array_values($deduped);
  }

}
