<?php

declare(strict_types=1);

namespace Drupal\component_lock\FormDecorator;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\form_decorator\Attribute\FormDecorator;
use Drupal\form_decorator\FormDecoratorBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation for layout builder block forms.
 */
#[FormDecorator('hook_form_alter')]
final class BlockComponentFormAlter extends FormDecoratorBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a BlockComponentFormAlter object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected AccountProxyInterface $currentUser,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function applies(): bool {
    return in_array($this->inner->getFormId(), [
      'layout_builder_add_block',
      'layout_builder_update_block',
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ...$args) {
    $form = parent::buildForm($form, $form_state, ...$args);
    $component = $this->getComponent($form_state);

    if ($this->currentUser->hasPermission('administer blocks')) {
      $form['lock_settings'] = [
        '#type' => 'details',
        '#title' => $this->t('Lock settings'),
        '#group' => 'tabs',
      ];
      $form['lock_settings']['lock_all_settings'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Lock all settings'),
        '#default_value' => $component->get('lock_all_settings') ?? $component->get('hide_settings'),
      ];
      $form['translation_settings'] = [
        '#type' => 'details',
        '#title' => $this->t('Translation settings'),
        '#group' => 'tabs',
      ];
      $form['translation_settings']['label_interface_translation'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Use interface translation for the label'),
        '#default_value' => $component->get('label_interface_translation'),
      ];

      $options = [];
      if (isset($form['settings'])) {
        foreach ($form['settings'] as $key => $element) {
          if (str_starts_with($key, '#')) {
            continue;
          }

          // Skip elements that are already invisible or not intended to be
          // toggled by the user.
          if (isset($element['#access']) && $element['#access'] === FALSE) {
            continue;
          }
          if (isset($element['#type']) && in_array($element['#type'], ['hidden', 'value'])) {
            continue;
          }

          $options[$key] = $element['#title'] ?? $key;
        }
      }

      if (!empty($options)) {
        $form['lock_settings']['locked_elements'] = [
          '#type' => 'checkboxes',
          '#title' => $this->t('Lock single elements'),
          '#options' => $options,
          '#default_value' => $component->get('locked_elements') ?: ($component->get('hidden_elements') ?: []),
          '#states' => [
            'visible' => [
              ':input[name="lock_settings[lock_all_settings]"]' => ['checked' => FALSE],
            ],
          ],
        ];
      }
    }

    // Lock the settings tab if the user does not have permission to edit
    // blocks.
    if ($component->get('lock_all_settings') || $component->get('hide_settings')) {
      $form['settings']['#access'] = $this->currentUser->hasPermission('administer blocks');
    }

    $locked_elements = $component->get('locked_elements') ?: ($component->get('hidden_elements') ?: []);
    if (!empty($locked_elements) && !$this->currentUser->hasPermission('administer blocks')) {
      foreach ($locked_elements as $locked_element) {
        if (isset($form['settings'][$locked_element])) {
          $form['settings'][$locked_element]['#access'] = FALSE;
        }
      }
    }

    // Workaround for a core bug where checkboxes with #return_value (like
    // label_display) lose their value when #access is FALSE.
    if (isset($form['settings']['label_display']) && isset($form['settings']['label_display']['#access']) && $form['settings']['label_display']['#access'] === FALSE) {
      $form['settings']['label_display']['#type'] = 'value';
      $form['settings']['label_display']['#value'] = $component->get('configuration')['label_display'] ?? NULL;
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $component = $this->getComponent($form_state);

    // Store the settings on the section additional storage.
    $component->set('lock_all_settings', $form_state->getValue(['lock_settings', 'lock_all_settings']));
    $component->set('label_interface_translation', $form_state->getValue(['translation_settings', 'label_interface_translation']));
    $locked_elements = array_filter($form_state->getValue(['lock_settings', 'locked_elements']) ?: []);
    $component->set('locked_elements', array_keys($locked_elements));

    return parent::submitForm($form, $form_state);
  }

}
