<?php

declare(strict_types=1);

namespace Drupal\conditions_field\Plugin\Field\FieldType;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\Core\Field\Attribute\FieldType;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Plugin\FilteredPluginManagerInterface;
use Drupal\json_field\Plugin\Field\FieldType\NativeJsonItem;

/**
 * Defines the 'conditions' field type.
 */
#[FieldType(
  id: "conditions",
  label: new TranslatableMarkup("Conditions"),
  description: new TranslatableMarkup("Stores condition configurations as json."),
  category: "json_data",
  default_widget: "conditions_groups",
  default_formatter: "conditions_empty_formatter",
  constraints: ["valid_json" => []],
)]
class ConditionsItem extends NativeJsonItem implements ConditionsItemInterface {

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    $schema = parent::schema($field_definition);
    $schema['columns']['condition_logic'] = [
      'description' => 'The condition logic for the conditions.',
      'type' => 'varchar',
      'length' => 255,
      'binary' => FALSE,
    ];
    $schema['columns']['group_condition_logic'] = [
      'description' => 'The condition logic for the conditions groups.',
      'type' => 'varchar',
      'length' => 255,
      'binary' => FALSE,
    ];
    $schema['columns']['status'] = [
      'description' => 'Whether the condition group is published or not.',
      'type' => 'int',
      'size' => 'tiny',
      'unsigned' => TRUE,
      'default' => 1,
    ];

    return $schema;
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    $value = $this->get('value')->getValue();
    return empty($value) || empty(JSON::decode($value));
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties = parent::propertyDefinitions($field_definition);
    $properties['condition_logic'] = DataDefinition::create('string')
      ->setLabel(t('The condition logic'));
    $properties['group_condition_logic'] = DataDefinition::create('string')
      ->setLabel(t('The group condition logic'));
    $properties['status'] = DataDefinition::create('integer')
      ->setLabel(t('The status'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public function preSave() {
    // Since not fields are already used, make sure some default values are set.
    if (!isset($this->group_condition_logic)) {
      $this->group_condition_logic = ConditionsItemInterface::CONDITIONS_GROUP_OPERATOR;
    }
    if (!isset($this->status)) {
      $this->status = ConditionsItemInterface::CONDITIONS_STATUS;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getConditions(): array {
    return JSON::decode($this->value) ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getConditionsOperator(): string {
    return $this->condition_logic ?? 'and';
  }

  /**
   * {@inheritdoc}
   */
  public function getConditionsGroupOperator(): string {
    return $this->group_condition_logic ?? 'or';
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus(): int {
    return (int) $this->status;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    return [
      'allowed_conditions' => [],
      'negate_allowed_conditions' => TRUE,
    ] + parent::defaultFieldSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::fieldSettingsForm($form, $form_state);
    $settings = $this->getSettings();

    $conditions_default_value = $options = [];
    foreach ($this->getConditionPluginDefinitions() as $id => $definition) {
      $options[$id] = $definition['label'];
      $allowed_conditions = $settings['allowed_conditions'] ?? $this->defaultFieldSettings()['allowed_conditions'];
      $conditions_default_value[$id] = match (in_array($id, $allowed_conditions)) {
        TRUE => $id,
        default => 0,
      };
    }
    
    $form['allowed_conditions'] = [
      '#title' => 'Allowed conditions',
      '#type' => 'checkboxes',
      '#options' => $options,
      '#default_value' => $conditions_default_value,
      '#element_validate' => [[static::class, 'validateConditions']],
    ];
    $form['negate_allowed_conditions'] = [
      '#title' => 'Negate allowed conditions',
      '#type' => 'checkbox',
      '#description' => $this->t('Using this option will convert the condition selection to a blacklist.'),
      '#default_value' => $settings['negate_allowed_conditions'] ?? $this->defaultFieldSettings()['negate_allowed_conditions'],
    ];
    return $form;
  }

  /**
   * Get condition plugin definitions.
   *
   * @return
   *   The condition plugin definitions
   */
  protected function getConditionPluginDefinitions(): array {
     return \Drupal::service('plugin.manager.condition')->getFilteredDefinitions('conditions_field');
  }

  /**
   * Use the same logic as TextItemBase::validateAllowedFormats.
   */
  public static function validateConditions(array &$element, FormStateInterface $form_state) {
    $value = array_values(array_filter($form_state->getValue($element['#parents'])));
    $form_state->setValueForElement($element, $value);
  }

}
