import { expect, test } from 'vitest';
import { fireEvent, render, screen, waitFor } from '@testing-library/react';
import { userEvent } from '@testing-library/user-event';
import App from './App.js';

describe('App', () => {
  beforeEach(() => {
    Object.defineProperty(window, 'drupalSettings', {
      writable: true,
      configurable: true,
      value: {
        conductor: {
          accountId: '12345',
          draftId: '',
        },
        canvas: {
          entityId: '1',
          entityType: 'canvas_page',
        },
      },
    });
  });

  test('4 tabs are present and first tab content is visible', () => {
    render(<App />);

    // Verify all 4 tabs are present
    const keywordsTab = screen.getByRole('tab', { name: /keywords/i });
    const competitorsTab = screen.getByRole('tab', { name: /competitors/i });
    // const guidanceTab = screen.getByRole('tab', { name: /guidance/i });
    const contentOutlineTab = screen.getByRole('tab', { name: /content outline/i });

    expect(keywordsTab).toBeInTheDocument();
    expect(competitorsTab).toBeInTheDocument();
    // expect(guidanceTab).toBeInTheDocument();
    expect(contentOutlineTab).toBeInTheDocument();

    // Verify the first tab (Keywords) is selected/active by default
    expect(keywordsTab).toHaveAttribute('aria-selected', 'true');
  });

  test('Keywords are saved to state', async () => {
    const user = userEvent.setup();

    render(<App />);

    const input = screen.getByRole('textbox');

    await user.type(input, 'jawn{Enter}');
    await user.type(input, 'tastykake{Enter}');
    expect(screen.queryByText(/^jawn$/)).toBeInTheDocument();
    expect(screen.queryByText(/^tastykake$/)).toBeInTheDocument();

    const competitorsTab = screen.getByRole('tab', { name: /competitors/i });
    await user.click(competitorsTab);

    expect(screen.queryByText(/^jawn$/)).not.toBeInTheDocument();
    expect(screen.queryByText(/^tastykake$/)).not.toBeInTheDocument();

    const keywordsTab = screen.getByRole('tab', { name: /keywords/i });
    await user.click(keywordsTab);

    expect(screen.queryByText(/^jawn$/)).toBeInTheDocument();
    expect(screen.queryByText(/^tastykake$/)).toBeInTheDocument();
  });

  test('Rank source is preserved', async () => {
    const user = userEvent.setup();
    const { container } = render(<App />);

    let singleValueElement = container.querySelector('.rank-source-selector__single-value');
    expect(singleValueElement).toHaveTextContent('Google (US / English)');

    fireEvent.mouseDown(container.querySelector('.rank-source-selector__dropdown-indicator')!, {
      button: 0,
    });

    await user.keyboard('United Kingdom');
    await user.keyboard('{Enter}');

    singleValueElement = container.querySelector('.rank-source-selector__single-value');
    expect(singleValueElement).toHaveTextContent('Google (United Kingdom / English)');

    const competitorsTab = screen.getByRole('tab', { name: /competitors/i });
    await user.click(competitorsTab);
    const keywordsTab = screen.getByRole('tab', { name: /keywords/i });
    await user.click(keywordsTab);

    singleValueElement = container.querySelector('.rank-source-selector__single-value');
    expect(singleValueElement).toHaveTextContent('Google (United Kingdom / English)');
  });

  test('Competitor URLs fetched when clicking the Generate Content Guidance button', async () => {
    const user = userEvent.setup();

    render(<App />);

    const input = screen.getByRole('textbox');

    await user.type(input, 'jawn{Enter}');
    await user.type(input, 'tastykake{Enter}');
    expect(screen.queryByText(/^jawn$/)).toBeInTheDocument();
    expect(screen.queryByText(/^tastykake$/)).toBeInTheDocument();

    const generateButton = screen.getByRole('button', { name: /generate content guidance/i });
    await user.click(generateButton);
    await waitFor(() => {
      expect(screen.queryByText('Loading')).not.toBeInTheDocument();
    });

    const listItems = screen.getAllByRole('listitem');
    expect(listItems).toHaveLength(30);
  });

  test('Competitor URLs fetched when changing tabs', async () => {
    const user = userEvent.setup();

    render(<App />);

    const input = screen.getByRole('textbox');

    await user.type(input, 'jawn{Enter}');
    await user.type(input, 'tastykake{Enter}');

    const competitorsTab = screen.getByRole('tab', { name: /competitors/i });
    await user.click(competitorsTab);
    await waitFor(() => {
      expect(screen.queryByText('Loading')).not.toBeInTheDocument();
    });

    const listItems = screen.getAllByRole('listitem');
    expect(listItems).toHaveLength(30);

    const label = screen.getByLabelText('https://en.wikipedia.org/wiki/Jawn');
    expect(label).toBeInTheDocument();
  });

  test('Competitor URLs updated when keywords are updated', async () => {
    const user = userEvent.setup();

    const { container } = render(<App />);

    let input = screen.getByRole('textbox');

    await user.type(input, 'jawn{Enter}');
    await user.type(input, 'tastykake{Enter}');

    const competitorsTab = screen.getByRole('tab', { name: /competitors/i });
    await user.click(competitorsTab);
    await waitFor(() => {
      expect(screen.queryByText('Loading')).not.toBeInTheDocument();
    });

    let firstLabel = container.querySelector('label:first-of-type');
    expect(firstLabel).toHaveTextContent('https://en.wikipedia.org/wiki/Jawn');

    const keywordsTab = screen.getByRole('tab', { name: /keywords/i });
    await user.click(keywordsTab);
    const removeJawnButton = screen.getByLabelText('Remove jawn');
    await user.click(removeJawnButton);
    const removeTastykakeButton = screen.getByLabelText('Remove tastykake');
    await user.click(removeTastykakeButton);
    expect(screen.queryByText('jawn')).not.toBeInTheDocument();
    expect(screen.queryByText('tastykake')).not.toBeInTheDocument();
    input = screen.getByRole('textbox');
    await user.type(input, 'gritty{Enter}');
    await user.type(input, 'phanatic{Enter}');
    await user.type(input, 'swoop{Enter}');

    const generateButton = screen.getByRole('button', { name: /generate content guidance/i });
    await user.click(generateButton);
    await waitFor(() => {
      expect(screen.queryByText('Loading')).not.toBeInTheDocument();
    });
    firstLabel = container.querySelector('label:first-of-type');
    expect(firstLabel).toHaveTextContent('https://www.instagram.com/grittynhl/');
  });

  test('Excluded competitor URLs are preserved', async () => {
    const user = userEvent.setup();

    render(<App />);

    const input = screen.getByRole('textbox');

    await user.type(input, 'jawn{Enter}');
    await user.type(input, 'tastykake{Enter}');

    const generateButton = screen.getByRole('button', { name: /generate content guidance/i });
    await user.click(generateButton);
    await waitFor(() => {
      expect(screen.queryByText('Loading')).not.toBeInTheDocument();
    });

    const listItems = screen.getAllByRole('listitem');
    expect(listItems).toHaveLength(30);

    const checkboxes = screen.getAllByRole('checkbox') as HTMLInputElement[];
    await user.click(checkboxes[0]!);
    await user.click(checkboxes[1]!);
    await user.click(checkboxes[2]!);
    const checkedCheckboxes = checkboxes.filter((checkbox) => checkbox.checked);
    expect(checkedCheckboxes).toHaveLength(27);

    const keywordsTab = screen.getByRole('tab', { name: /keywords/i });
    await user.click(keywordsTab);
    // const guidanceTab = screen.getByRole('tab', { name: /guidance/i });
    // await user.click(guidanceTab);
    const competitorsTab = screen.getByRole('tab', { name: /competitors/i });
    await user.click(competitorsTab);

    const checkboxesAgain = screen.getAllByRole('checkbox') as HTMLInputElement[];
    const checkedCheckboxesAgain = checkboxesAgain.filter((checkbox) => checkbox.checked);
    expect(checkedCheckboxesAgain).toHaveLength(27);
  });
});
