import { describe, test, expect, vi } from 'vitest';
import { render, screen, waitFor } from '@testing-library/react';
import { userEvent } from '@testing-library/user-event';
import ContentGeneratorTextArea from './ContentGeneratorTextArea.js';

// Mock ResizeObserver
global.ResizeObserver = vi.fn().mockImplementation(() => ({
  observe: vi.fn(),
  unobserve: vi.fn(),
  disconnect: vi.fn(),
}));

vi.mock('../../assets/lightbulb.svg?react', () => ({
  default: () => <div data-testid="lightbulb-icon">Lightbulb</div>,
}));

describe('ContentGeneratorTextArea', () => {
  test('renders lightbulb button after editor toolbar mounts', async () => {
    render(<ContentGeneratorTextArea draftId="12345" />);

    await waitFor(() => {
      expect(screen.getByLabelText('Generate content with AI')).toBeInTheDocument();
    });
  });

  test('opens menu when lightbulb is clicked', async () => {
    const user = userEvent.setup();
    const { container } = render(<ContentGeneratorTextArea draftId="12345" />);

    // Wait for CKEditor toolbar
    await waitFor(() => {
      expect(container.querySelector('.ck-editor__top')).toBeInTheDocument();
    });

    // Give MutationObserver time to run
    await new Promise((resolve) => setTimeout(resolve, 500));

    const lightbulb = await screen.findByLabelText('Generate content with AI');

    await user.click(lightbulb);

    expect(screen.getByTestId('content-generator-text-area__context-menu')).toBeInTheDocument();
    expect(screen.getByText('Replace with generated outline')).toBeInTheDocument();
    expect(screen.getByText('Replace with generated draft')).toBeInTheDocument();
  });

  test('closes menu when lightbulb is clicked again', async () => {
    const user = userEvent.setup();
    const { container } = render(<ContentGeneratorTextArea draftId="12345" />);

    // Wait for CKEditor toolbar
    await waitFor(() => {
      expect(container.querySelector('.ck-editor__top')).toBeInTheDocument();
    });

    // Give MutationObserver time to run
    await new Promise((resolve) => setTimeout(resolve, 500));

    const lightbulb = await screen.findByLabelText('Generate content with AI');

    await user.click(lightbulb);
    expect(screen.getByText('Replace with generated outline')).toBeInTheDocument();
    await user.click(lightbulb);
    expect(screen.queryByText('Replace with generated outline')).not.toBeInTheDocument();
  });

  test('displays streaming overlay when loading', async () => {
    const user = userEvent.setup();
    const { container } = render(<ContentGeneratorTextArea draftId="12345" />);

    // Wait for CKEditor toolbar
    await waitFor(() => {
      expect(container.querySelector('.ck-editor__top')).toBeInTheDocument();
    });

    // Give MutationObserver time to run
    await new Promise((resolve) => setTimeout(resolve, 500));

    const lightbulb = await screen.findByLabelText('Generate content with AI');

    await user.click(lightbulb);
    const generateOutline = screen.getByText('Replace with generated outline');
    await user.click(generateOutline);

    const messageElement = screen.getByText(/Generating content/i);
    expect(messageElement).toBeInTheDocument();
    await waitFor(
      () => {
        expect(screen.queryByText(/Generating content/i)).not.toBeInTheDocument();
      },
      { timeout: 3000 },
    );
  });

  //test('generates text when "Generate outline" is clicked', async () => {
  //  const user = userEvent.setup();
  //  render(<ContentGeneratorTextArea draftId="12345" />);

  //  const lightbulb = await screen.findByLabelText('Generate content with AI');
  //  await user.click(lightbulb);
  //  const generateOutline = screen.getByText('Generate outline');
  //  await user.click(generateOutline);

  //  await waitFor(() => {
  //    const editableArea = screen.getByRole('textbox');
  //    expect(editableArea.textContent).toContain(
  //      'Strategic overview: This content focuses specifically on the unique cultural significance',
  //    );
  //  });
  //});

  //test('generates text when "Generate draft" is clicked', async () => {
  //  const user = userEvent.setup();
  //  render(<ContentGeneratorTextArea draftId="12345" />);

  //  const lightbulb = await screen.findByLabelText('Generate content with AI');
  //  await user.click(lightbulb);
  //  const generateDraft = screen.getByText('Generate draft');
  //  await user.click(generateDraft);

  //  await waitFor(() => {
  //    const editableArea = screen.getByRole('textbox');
  //    expect(editableArea.textContent).toContain(
  //      'Philadelphia sports culture is legendary for its passionate fans and compelling traditions',
  //    );
  //  });
  //});

  //test('closes menu after "Generate outline" is clicked', async () => {
  //  const user = userEvent.setup();
  //  render(<ContentGeneratorTextArea draftId="12345" />);

  //  const lightbulb = await screen.findByLabelText('Generate content with AI');
  //  await user.click(lightbulb);
  //  await user.click(screen.getByText('Generate outline'));

  //  expect(screen.queryByText('Generate outline')).not.toBeInTheDocument();
  //});

  //test('closes menu after "Generate draft" is clicked', async () => {
  //  const user = userEvent.setup();
  //  render(<ContentGeneratorTextArea draftId="12345" />);

  //  const lightbulb = await screen.findByLabelText('Generate content with AI');
  //  await user.click(lightbulb);
  //  await user.click(screen.getByText('Generate draft'));

  //  expect(screen.queryByText('Generate draft')).not.toBeInTheDocument();
  //});

  //test('calls cancel when Cancel button is clicked in streaming overlay', async () => {
  //  const user = userEvent.setup();
  //  render(<ContentGeneratorTextArea draftId="12345" />);

  //  const lightbulb = await screen.findByLabelText('Generate content with AI');
  //  await user.click(lightbulb);
  //  await user.click(screen.getByText('Generate outline'));

  //  const cancelButton = screen.getByRole('button', { name: 'Cancel' });
  //  await user.click(cancelButton);

  //  const editableArea = screen.getByRole('textbox');
  //  expect(editableArea.textContent).not.toContain(
  //    'enduring role in Philadelphia sports and fandom',
  //  );
  //});

  //test('clears editor text when "Clear text" is clicked', async () => {
  //  const user = userEvent.setup();
  //  render(<ContentGeneratorTextArea draftId="12345" />);

  //  const lightbulb = await screen.findByLabelText('Generate content with AI');
  //  await user.click(lightbulb);
  //  const generateOutline = screen.getByText('Generate outline');
  //  await user.click(generateOutline);

  //  await waitFor(() => {
  //    const editableArea = screen.getByRole('textbox');
  //    expect(editableArea.textContent).toContain('enduring role in Philadelphia sports and fandom');
  //  });

  //  await user.click(lightbulb);
  //  const clearText = screen.getByText('Clear text');
  //  await user.click(clearText);
  //  const editableArea = screen.getByRole('textbox');
  //  const textContent = editableArea.textContent?.replace(/[\u2060\u200B-\u200D\uFEFF\s]/g, '');
  //  expect(textContent).toBe('');
  //});

  //test('displays error overlay when there is an error', async () => {
  //  const user = userEvent.setup();
  //  render(<ContentGeneratorTextArea draftId="12345-bad" />);

  //  const lightbulb = await screen.findByLabelText('Generate content with AI');
  //  await user.click(lightbulb);
  //  const generateOutline = screen.getByText('Generate outline');
  //  await user.click(generateOutline);

  //  expect(screen.getByText('Error')).toBeInTheDocument();
  //  expect(screen.getByText(/There was an error generating content/)).toBeInTheDocument();
  //  expect(screen.getByRole('button', { name: 'Close' })).toBeInTheDocument();
  //});

  //test('calls cancel when Close button is clicked in error overlay', async () => {
  //  const user = userEvent.setup();
  //  render(<ContentGeneratorTextArea draftId="12345-bad" />);

  //  const lightbulb = await screen.findByLabelText('Generate content with AI');
  //  await user.click(lightbulb);
  //  const generateOutline = screen.getByText('Generate outline');
  //  await user.click(generateOutline);
  //  const cancelButton = screen.getByRole('button', { name: 'Close' });
  //  await user.click(cancelButton);
  //  expect(screen.queryByText('Error')).not.toBeInTheDocument();
  //});

  //test('applies streaming class when loading', async () => {
  //  const user = userEvent.setup();
  //  const { container } = render(<ContentGeneratorTextArea draftId="abc123" />);

  //  const lightbulb = await screen.findByLabelText('Generate content with AI');
  //  await user.click(lightbulb);
  //  const generateOutline = screen.getByText('Generate outline');
  //  await user.click(generateOutline);

  //  const mainDiv = container.querySelector('.content-generator-text-area');
  //  expect(mainDiv).toHaveClass('content-generator-text-area--is-streaming');
  //});

  //test('does not apply streaming class when error occurs', async () => {
  //  const user = userEvent.setup();
  //  const { container } = render(<ContentGeneratorTextArea draftId="12345-bad" />);

  //  const lightbulb = await screen.findByLabelText('Generate content with AI');
  //  await user.click(lightbulb);
  //  const generateOutline = screen.getByText('Generate outline');
  //  await user.click(generateOutline);

  //  const mainDiv = container.querySelector('.content-generator-text-area');
  //  expect(mainDiv).not.toHaveClass('content-generator-text-area--is-streaming');
  //});
});
