import React from 'react';
import { describe, test, expect } from 'vitest';
import { fireEvent, render, waitFor, screen } from '@testing-library/react';
import { userEvent } from '@testing-library/user-event';
import RankSourceSelector, { type RankSource } from './RankSourceSelector.js';

describe('RankSourceSelector Component', () => {
  test('Renders all options', async () => {
    const { container } = render(<RankSourceSelector accountId={'12345'} />);

    fireEvent.mouseDown(container.querySelector('.rank-source-selector__dropdown-indicator')!, {
      button: 0,
    });

    await waitFor(() => {
      const menu = document.querySelector('.rank-source-selector__menu');
      expect(menu).toBeInTheDocument();
      expect(menu).toBeVisible();
    });

    await waitFor(() =>
      expect(
        container.querySelectorAll('.rank-source-selector__option[id*="-option-"]'),
      ).toHaveLength(223),
    );
  });

  test('Selects the option passed as a prop', async () => {
    const rankSource = {
      id: 1,
      description: 'Google (US / English)',
      isoLocale: 'en_US',
    };
    const setRankSource = vi.fn();
    render(
      <RankSourceSelector
        accountId={'12345'}
        rankSource={rankSource}
        setRankSource={setRankSource}
      />,
    );

    const selectedValue = await screen.findByText('Google (US / English)');
    expect(selectedValue).toBeInTheDocument();
  });

  test('External state is updated', async () => {
    const user = userEvent.setup();
    const rankSource = {};
    const setRankSource = vi.fn();
    const { container } = render(
      <RankSourceSelector
        accountId={'12345'}
        rankSource={rankSource}
        setRankSource={setRankSource}
      />,
    );

    fireEvent.mouseDown(container.querySelector('.rank-source-selector__dropdown-indicator')!, {
      button: 0,
    });
    await user.click(screen.getByText('Google (Angola / English)'));

    expect(setRankSource).toHaveBeenCalledWith({
      id: 175,
      description: 'Google (Angola / English)',
      isoLocale: 'en_AO',
    });
  });

  test('Given an ID, description and isoLocale are populated', async () => {
    let rankSource: RankSource = {
      id: 1,
    };
    const setRankSource = (newRankSource: RankSource) => (rankSource = newRankSource);
    render(
      <RankSourceSelector
        accountId={'12345'}
        rankSource={rankSource}
        setRankSource={setRankSource}
      />,
    );

    expect(rankSource).toStrictEqual({
      id: 1,
      description: 'Google (US / English)',
      isoLocale: 'en_US',
    });
  });
});
