import React, { useEffect } from 'react';
import {
  useKeywords,
  useCompetitorUrls,
  useExcludedCompetitorUrls,
  useAppStateDispatch,
  useAccountId,
  useRankSource,
} from '../Context.js';
import useSWR from 'swr';
import qs from 'qs';
import { ExternalLink } from 'lucide-react';
import { truncateText } from '../utilities/text.js';
import { get } from '../utilities/fetcher.js';
import Button from '../components/Button.js';
import ErrorMessage from '../components/ErrorMessage.js';
import Loading from '../components/Loading.js';
import InfoBar from '../components/InfoBar.js';
import './Competitors.css';

export interface Serp {
  rank: number;
  url: string;
  keyword: string;
  label: string;
}

const Competitors = ({
  setTab = (_: number) => {},
  nextTab = 0,
}: {
  setTab?: (index: number) => void;
  nextTab?: number;
}) => {
  const keywords = useKeywords();

  if (!keywords.length) {
    return (
      <div>
        <InfoBar>
          Keywords are required, please go back to the previous screen to select some.
        </InfoBar>
      </div>
    );
  }

  const competitorUrls = useCompetitorUrls();
  const excludedCompetitorUrls = useExcludedCompetitorUrls();
  const dispatch = useAppStateDispatch();

  const accountId = useAccountId();
  const rankSource = useRankSource();

  const params = {
    queries: keywords,
    first: 15,
    rankSourceId: rankSource.id,
  };

  const { data, isLoading, error, mutate } = useSWR(
    `/conductor/proxy/v3/accounts/${accountId}/batch-untracked-serp/serp-explorer?${qs.stringify(params)}`,
    get,
    { fallbackData: [] },
  );
  useEffect(() => {
    if (data?.length) {
      const urls = data.map((serp: Serp) => serp.url);

      // This was previously loaded, either as an existing draft or earlier in the application
      // flow, so any new URLs fetched after this should be added to excluded URLs.
      if (competitorUrls?.length) {
        const newExcludedUrls = urls.filter(
          (url: string) => !competitorUrls.includes(url) && !excludedCompetitorUrls.includes(url),
        );
        dispatch({
          type: 'setExcludedCompetitorUrls',
          excludedCompetitorUrls: [...excludedCompetitorUrls, ...newExcludedUrls],
        });
      } else {
        dispatch({
          type: 'setCompetitorUrls',
          competitorUrls: urls,
        });
      }
    }
  }, [data]);

  const handleCheckboxChange = (serp: Serp, checked: boolean) => {
    if (checked) {
      dispatch({
        type: 'setExcludedCompetitorUrls',
        excludedCompetitorUrls: excludedCompetitorUrls.filter(
          (unselectedSerp) => unselectedSerp !== serp.url,
        ),
      });
      dispatch({
        type: 'setCompetitorUrls',
        competitorUrls: [...competitorUrls, serp.url],
      });
    } else {
      dispatch({
        type: 'setExcludedCompetitorUrls',
        excludedCompetitorUrls: [...excludedCompetitorUrls, serp.url],
      });
      dispatch({
        type: 'setCompetitorUrls',
        competitorUrls: competitorUrls.filter((unselectedSerp) => unselectedSerp !== serp.url),
      });
    }
  };

  return (
    <div className="competitors">
      <div className="competitors__description">
        Content guidance comes from top competitors’ pages ranking for your keywords. Remove
        irrelevant ones for better results.
      </div>
      <InfoBar>A minimum of 10 competitors is required.</InfoBar>
      <ul className="competitors__list">
        {isLoading && (
          <div className="competitors__loading">
            <Loading />
          </div>
        )}
        {error && (
          <div className="competitors__error">
            <ErrorMessage>
              Something went wrong.{' '}
              <button
                onClick={() => {
                  mutate();
                }}
              >
                Please try again.
              </button>
            </ErrorMessage>
          </div>
        )}
        {data.map((serp: Serp) => (
          <li key={serp.url}>
            <label className="competitors__item">
              <input
                type="checkbox"
                checked={!excludedCompetitorUrls.includes(serp.url)}
                onChange={(e) => handleCheckboxChange(serp, e.target.checked)}
              />
              <a
                className="competitors__external"
                href={serp.url}
                target="_blank"
                rel="noopener noreferrer"
                title={serp.url}
              >
                {truncateText(serp.url)} <ExternalLink size={16} />
              </a>
            </label>
          </li>
        ))}
      </ul>
      <div className="competitors__generate-guidance">
        <Button disabled={competitorUrls.length < 10} onClick={() => setTab(nextTab)}>
          Generate content guidance
        </Button>
      </div>
    </div>
  );
};

export default Competitors;
