import React, { useState } from 'react';
import CreateTitle from '../components/CreateTitle.js';
import Loading from '../components/Loading.js';
import InfoBar from '../components/InfoBar.js';
import ErrorMessage from '../components/ErrorMessage.js';
import DraftManager from '../utilities/DraftManager.js';
import ContentGeneratorTextArea from '../components/ContentGeneratorTextArea.js';
import {
  useAppStateDispatch,
  useDraft,
  useCompetitorUrls,
  useKeywords,
  useAccountId,
  useBrand,
} from '../Context.js';
import CreateMetaDescription from '../components/CreateMetaDescription.js';
import './ContentOutline.css';

const ContentOutline = ({}: { setTab?: (index: number) => void; nextTab?: number }) => {
  const dispatch = useAppStateDispatch();
  const accountId = useAccountId();
  const brand = useBrand();
  const draft = useDraft();
  const keywords = useKeywords();
  const competitorUrls = useCompetitorUrls();
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(false);

  if (!keywords.length || competitorUrls.length < 10) {
    return (
      <div>
        <InfoBar>
          Keywords and at least 10 competitor URLs are required, please go back to the previous
          screens to select some.
        </InfoBar>
      </div>
    );
  }

  return (
    <div
      className={`content-outline ${loading ? 'content-outline--loading' : ''} ${error ? 'content-outline--error' : ''}`}
    >
      <DraftManager
        isLoading={(loading) => setLoading(loading)}
        hasError={(error) => setError(error)}
      />
      {(error || (!loading && !draft.lastUpdated)) && (
        <ErrorMessage>
          There's been an error communicating with our systems. You can leave this window open
          whilst we try to re-establish a connection, or restart the application.
        </ErrorMessage>
      )}
      {loading && <Loading />}
      {!loading && draft.id && draft.lastUpdated && (
        <>
          <CreateTitle
            accountId={accountId}
            draftId={draft.id}
            brand={brand}
            setBrand={(text: string) =>
              dispatch({
                type: 'setBrand',
                brand: text,
              })
            }
            text={draft.title ?? ''}
            setText={(text: string) =>
              dispatch({
                type: 'updateDraft',
                draft: {
                  title: text,
                },
              })
            }
          />
          <CreateMetaDescription
            accountId={accountId}
            draftId={draft.id}
            text={draft.metaDescription ?? ''}
            setText={(text: string) =>
              dispatch({
                type: 'updateDraft',
                draft: {
                  metaDescription: text,
                },
              })
            }
          />
          <ContentGeneratorTextArea
            draftId={draft.id}
            initialText={draft.bodyCopy || ''}
            setText={(text: string) =>
              dispatch({
                type: 'updateDraft',
                draft: {
                  bodyCopy: text,
                },
              })
            }
          />
        </>
      )}
    </div>
  );
};

export default ContentOutline;
