<?php

declare(strict_types=1);

namespace Drupal\Tests\conductor\Functional;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\Tests\ApiRequestTrait;
use Drupal\Tests\conductor\Trait\DraftTrait;
use PHPUnit\Framework\Attributes\Group;

/**
 * Test for the Draft api.
 */
#[Group("conductor")]
final class ConductorDraftDashboardTest extends ConductorBrowserTestBase {

  use ApiRequestTrait;
  use DraftTrait;

  private const MAX_DRAFTS = 10;
  private TimeInterface $time;
  private DateFormatterInterface $dateFormatter;
  private int $currentYear;

  /**
   * {@inheritdoc}
   *
   * Enable only the module under test for now.
   */
  protected static $modules = [
    'conductor',
    'conductor_test',
    'node',
    'canvas',
  ];

  /**
   * {@inheritdoc}
   *
   * Use a minimal core theme to avoid unrelated dependencies.
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();
    $this->container->get('config.factory')->getEditable('conductor.settings')->set('max_drafts', self::MAX_DRAFTS)->save();
    $time = $this->container->get('datetime.time');
    \assert($time instanceof TimeInterface);
    $this->time = $time;
    $dateFormatter = $this->container->get('date.formatter');
    \assert($dateFormatter instanceof DateFormatterInterface);
    $this->dateFormatter = $dateFormatter;
    $this->currentYear = (int) $this->dateFormatter->format($this->time->getCurrentTime(), 'custom', 'Y');

    $this->drupalCreateContentType([
      'type' => 'article',
      'name' => 'Article',
    ]);
  }

  public function testDashboardWithoutKey(): void {
    for ($i = 0; $i < 5; $i++) {
      $this->createHistoricalDraft($this->currentYear);
    }
    $url = Url::fromRoute('conductor.draft_dashboard');
    $conductor_user = $this->createUser(['use conductor']);
    \assert($conductor_user instanceof AccountInterface);
    $this->drupalLogin($conductor_user);
    $this->drupalGet($url);
    // There should not be any drafts for this node.
    $this->assertSession()->statusCodeEquals(200);
  }

  public function testRolesToVisitDraftDashboard(): void {
    $this->generateKey();
    $url = Url::fromRoute('conductor.draft_dashboard');
    $anonymous_user = $this->createUser(['access content']);
    \assert($anonymous_user instanceof AccountInterface);
    $this->drupalLogin($anonymous_user);
    $this->drupalGet($url);
    // Anonymous users should not have access to the draft API.
    $this->assertSession()->statusCodeEquals(403);

    $conductor_user = $this->createUser(['use conductor']);
    \assert($conductor_user instanceof AccountInterface);
    $this->drupalLogin($conductor_user);
    $this->drupalGet($url);
    // We should have no error.
    $this->assertSession()->statusCodeEquals(200);
    // But as there are no drafts, expect the empty text.
    $this->assertSession()->responseContains('No drafts found for the current year');
  }

  public function testDashboardWithDrafts(): void {
    $this->generateKey();
    $url = Url::fromRoute('conductor.draft_dashboard');
    $conductor_user = $this->createUser(['use conductor']);
    \assert($conductor_user instanceof AccountInterface);
    for ($i = 0; $i < 5; $i++) {
      $this->createHistoricalDraft($this->currentYear);
    }
    $this->createHistoricalDraft(2019);
    $this->createHistoricalDraft(2019);
    $this->createHistoricalDraft(2019);
    $this->drupalLogin($conductor_user);
    $this->drupalGet($url);
    // It should only show the 5 drafts created for the current year.
    $this->assertSession()->elementsCount('css', 'table.drafts tbody tr', 5);

    $this->assertSession()->pageTextContains("5/" . self::MAX_DRAFTS . " drafts used for year {$this->currentYear}");
  }

}
