import React, { useState, useEffect } from 'react';
import Select, { type ControlProps, components } from 'react-select';
import { Globe } from 'lucide-react';
import useSWRImmutable from 'swr/immutable';
import { get } from '../utilities/fetcher.js';
import './RankSourceSelector.css';

export interface RankSource {
  id?: number;
  description?: string;
  isoLocale?: string;
}

interface SelectOption {
  value: string;
  label: string;
}

const RankSourceSelector = ({
  accountId,
  rankSource = {},
  setRankSource = (_) => {},
}: {
  accountId: string;
  rankSource?: RankSource;
  setRankSource?: (_: RankSource) => void;
}) => {
  const [selected, setSelected] = useState({ value: '', label: '' });

  const { data: rankSourcesData, isLoading } = useSWRImmutable<RankSource[]>(
    `/conductor/proxy/v3/${accountId}/rank-sources/content-guidance`,
    get,
  );

  const rankSourcesById =
    rankSourcesData?.reduce((acc: { [key: string]: RankSource }, val: RankSource) => {
      const key = `${val.id}`;
      if (val.id && val.description && val.isoLocale) {
        acc[key] = {
          id: val.id,
          description: val.description,
          isoLocale: val.isoLocale,
        };
      }
      return acc;
    }, {}) || {};

  // Add missing description and isoLocale if this was loaded from a draft as only
  // the id is saved.
  useEffect(() => {
    if (
      rankSourcesData?.length &&
      rankSource.id &&
      (!rankSource.description || !rankSource.isoLocale)
    ) {
      const rankSourceData = rankSourcesData
        .filter((rankSourceData) => rankSourceData.id === rankSource.id)
        .pop();
      if (rankSourceData?.description && rankSourceData?.isoLocale) {
        setRankSource({
          id: rankSource.id,
          description: rankSourceData.description,
          isoLocale: rankSourceData.isoLocale,
        });
      }
    }
  }, [rankSourcesData]);

  useEffect(() => {
    if (selected.value && rankSourcesById[selected.value]) {
      const selectedRankSource = rankSourcesById[selected.value];
      if (selectedRankSource) {
        setRankSource(selectedRankSource);
      }
    }
  }, [selected]);

  const selectOptions: SelectOption[] =
    rankSourcesData?.map((rankSource) => ({
      value: `${rankSource.id}`,
      label: rankSource.description ?? '',
    })) ?? [];

  let value: SelectOption | null = null;
  if (!isLoading && rankSource.id && rankSource.description) {
    value = {
      value: `${rankSource.id}`,
      label: rankSource.description,
    };
  }

  const SelectControl = (props: ControlProps<{ value: string; label: string }, false>) => (
    <components.Control {...props}>
      <Globe size={16} style={{ marginRight: '8px', color: '#6b7280' }} />
      {props.children}
    </components.Control>
  );

  return (
    <Select
      onChange={(newValue) => {
        setSelected(newValue as SelectOption);
      }}
      value={value}
      options={selectOptions}
      components={{
        Control: SelectControl,
      }}
      className="rank-source-selector"
      classNamePrefix="rank-source-selector"
      unstyled
      isSearchable={true}
      isMulti={false}
      isDisabled={isLoading}
      placeholder={isLoading ? 'Loading rank sources…' : 'Select rank source'}
      aria-label="Choose a rank source option"
    />
  );
};

export default RankSourceSelector;
