import React, { createContext, useContext, useReducer, useMemo, type ReactNode } from 'react';
import { type Draft } from './utilities/types.js';

interface AppState {
  accountId: string;
  drupal: {
    entity: {
      type: string;
      id: string;
    };
  };
  keywords: string[];
  rankSource: {
    id: number;
    description: string;
    isoLocale: string;
  };
  competitorUrls: string[];
  excludedCompetitorUrls: string[];
  brand: string;
  draft: Draft;
}

type AppStateAction =
  | { type: 'setKeywords'; keywords: string[] }
  | { type: 'setRankSource'; rankSource: { id: number; description: string; isoLocale: string } }
  | { type: 'setCompetitorUrls'; competitorUrls: string[] }
  | { type: 'setExcludedCompetitorUrls'; excludedCompetitorUrls: string[] }
  | { type: 'setBrand'; brand: string }
  | { type: 'updateDraft'; draft: Draft };

type AppStateContextType = AppState | null;
type AppStateDispatchContextType = ((action: AppStateAction) => void) | null;

interface AppStateProviderProps {
  children: ReactNode;
  initialState?: Partial<AppState>; // Add this prop
}

const AppStateContext = createContext<AppStateContextType>(null);
const AppStateDispatchContext = createContext<AppStateDispatchContextType>(null);

export function AppStateProvider({ children, initialState }: AppStateProviderProps) {
  const mergedInitialState = {
    ...initialAppState,
    ...initialState, // Override with provided initial state
  };

  const [tasks, dispatch] = useReducer(appStateReducer, mergedInitialState);

  return (
    <AppStateContext.Provider value={tasks}>
      <AppStateDispatchContext.Provider value={dispatch}>
        {children}
      </AppStateDispatchContext.Provider>
    </AppStateContext.Provider>
  );
}

export function useAppState(): AppState {
  const context = useContext(AppStateContext);
  if (context === null) {
    throw new Error('useAppState must be used within a TasksProvider');
  }
  return context;
}

function appStateReducer(appState: AppState, action: AppStateAction): AppState {
  switch (action.type) {
    case 'setKeywords': {
      return {
        ...appState,
        keywords: action.keywords,
      };
    }
    case 'setRankSource': {
      return {
        ...appState,
        rankSource: action.rankSource,
      };
    }
    case 'setCompetitorUrls': {
      return {
        ...appState,
        competitorUrls: action.competitorUrls,
      };
    }
    case 'setExcludedCompetitorUrls': {
      return {
        ...appState,
        excludedCompetitorUrls: action.excludedCompetitorUrls,
      };
    }
    case 'setBrand': {
      return {
        ...appState,
        brand: action.brand,
      };
    }
    case 'updateDraft': {
      return {
        ...appState,
        draft: {
          ...appState.draft,
          ...action.draft,
        },
      };
    }
    default: {
      throw Error('Unknown action: ' + action);
    }
  }
}

const initialAppState: AppState = {
  accountId: '',
  drupal: {
    entity: {
      type: '',
      id: '',
    },
  },
  keywords: [],
  rankSource: {
    id: 1,
    description: 'Google (US / English)',
    isoLocale: 'en_US',
  },
  competitorUrls: [],
  excludedCompetitorUrls: [],
  brand: '',
  draft: {
    id: null,
    title: 'Untitled Draft',
  },
};

export function useAccountId(): string {
  const context = useContext(AppStateContext);
  if (context === null) {
    throw new Error('useAccountId must be used within an AppStateProvider');
  }
  return useMemo(() => context.accountId, [context.accountId]);
}

export function useDrupal(): { entity: { type: string; id: string } } {
  const context = useContext(AppStateContext);
  if (context === null) {
    throw new Error('useDrupal must be used within an AppStateProvider');
  }
  return useMemo(() => context.drupal, [context.drupal]);
}

export function useKeywords(): string[] {
  const context = useContext(AppStateContext);
  if (context === null) {
    throw new Error('useKeywords must be used within an AppStateProvider');
  }
  return useMemo(() => context.keywords, [context.keywords]);
}

export function useRankSource(): { id: number; description: string; isoLocale: string } {
  const context = useContext(AppStateContext);
  if (context === null) {
    throw new Error('useRankSource must be used within an AppStateProvider');
  }
  return useMemo(() => context.rankSource, [context.rankSource]);
}

export function useLocale(): string {
  const context = useContext(AppStateContext);
  if (context === null) {
    throw new Error('useLocale must be used within an AppStateProvider');
  }
  return useMemo(() => context.rankSource.isoLocale, [context.rankSource.isoLocale]);
}

export function useCompetitorUrls(): string[] {
  const context = useContext(AppStateContext);
  if (context === null) {
    throw new Error('useCompetitorUrls must be used within an AppStateProvider');
  }
  return useMemo(() => context.competitorUrls, [context.competitorUrls]);
}

export function useExcludedCompetitorUrls(): string[] {
  const context = useContext(AppStateContext);
  if (context === null) {
    throw new Error('useExcludedCompetitorUrls must be used within an AppStateProvider');
  }
  return useMemo(() => context.excludedCompetitorUrls, [context.excludedCompetitorUrls]);
}

export function useBrand(): string {
  const context = useContext(AppStateContext);
  if (context === null) {
    throw new Error('useBrand must be used within an AppStateProvider');
  }
  return useMemo(() => context.brand, [context.brand]);
}

export function useDraft(): Draft {
  const context = useContext(AppStateContext);
  if (context === null) {
    throw new Error('useDraft must be used within an AppStateProvider');
  }
  return useMemo(() => context.draft, [context.draft]);
}

export function useAppStateDispatch(): (action: AppStateAction) => void {
  const context = useContext(AppStateDispatchContext);
  if (context === null) {
    throw new Error('useTasksDispatch must be used within a TasksProvider');
  }
  return context;
}
