import { describe, test, expect, vi } from 'vitest';
import { render, screen, waitFor } from '@testing-library/react';
import { userEvent } from '@testing-library/user-event';

const dispatchSpy = vi.fn();
// Mock the module importing.
vi.mock('../Context.js', async () => {
  const actual = await vi.importActual('../Context.js');
  return {
    ...actual,
    useAppStateDispatch: () => dispatchSpy,
  };
});

import { AppStateProvider } from '../Context.js';
import CreateTitle from './CreateTitle.js';

vi.mock('../../assets/lightbulb.svg?react', () => ({
  default: () => <div data-testid="lightbulb-icon">Lightbulb</div>,
}));

describe('CreateTitle component', () => {
  test('renders the title tag label', () => {
    render(
      <AppStateProvider initialState={{ accountId: '12345' }}>
        <CreateTitle />
      </AppStateProvider>,
    );
    expect(screen.getByText('Title tag')).toBeInTheDocument();
  });

  test('displays Topic and Length validation sections', () => {
    render(
      <AppStateProvider initialState={{ accountId: '12345' }}>
        <CreateTitle />
      </AppStateProvider>,
    );
    expect(screen.getByText('Topic')).toBeInTheDocument();
    expect(screen.getByText('Length')).toBeInTheDocument();
  });

  test('renders the text input field', () => {
    render(
      <AppStateProvider initialState={{ accountId: '12345' }}>
        <CreateTitle />
      </AppStateProvider>,
    );
    const input = screen.getByTestId('title-tag-input');
    expect(input).toBeInTheDocument();
  });

  test('updates input value when typing', async () => {
    const user = userEvent.setup();
    render(
      <AppStateProvider initialState={{ accountId: '12345' }}>
        <CreateTitle />
      </AppStateProvider>,
    );

    const input = screen.getByTestId('title-tag-input');
    await user.type(input, 'My Test Title');

    expect(input).toHaveValue('My Test Title');
  });

  test('dispatches updateDraft action when input changes', async () => {
    const user = userEvent.setup();
    render(
      <AppStateProvider initialState={{ accountId: '12345' }}>
        <CreateTitle />
      </AppStateProvider>,
    );

    const input = screen.getByTestId('title-tag-input');
    await user.type(input, 'Test');

    // @todo fix this.
    //expect(dispatchSpy).toHaveBeenCalledWith({
    //  type: 'updateDraft',
    //  draft: { title: 'Test' },
    //});
  });

  test('opens the dropdown menu when open button is clicked', async () => {
    const user = userEvent.setup();
    const { container } = render(
      <AppStateProvider initialState={{ accountId: '12345' }}>
        <CreateTitle />
      </AppStateProvider>,
    );

    const openButton = container.querySelector(
      '.create-title__select__dropdown-indicator',
    ) as Element;
    await user.click(openButton);

    await waitFor(() => {
      expect(screen.getByRole('listbox')).toBeInTheDocument();
    });
  });

  test('displays generated title options in the menu', async () => {
    const user = userEvent.setup();
    const { container } = render(
      <AppStateProvider initialState={{ accountId: '12345' }}>
        <CreateTitle />
      </AppStateProvider>,
    );

    const openButton = container.querySelector(
      '.create-title__select__dropdown-indicator',
    ) as Element;
    await user.click(openButton);

    const loadingSpinner = screen.getByRole('status');
    expect(loadingSpinner).toBeInTheDocument();

    await waitFor(
      () => {
        const loadingSpinner = screen.queryByText(/loading/i);
        expect(loadingSpinner).not.toBeInTheDocument();
      },
      { timeout: 3000 },
    );

    expect(
      screen.getByText('Why Gritty, Phanatic & Swoop Define Philadelphia Sports'),
    ).toBeInTheDocument();
  });

  test('selects a title option when clicked', async () => {
    const user = userEvent.setup();
    const { container } = render(
      <AppStateProvider initialState={{ accountId: '12345' }}>
        <CreateTitle />
      </AppStateProvider>,
    );

    const openButton = container.querySelector(
      '.create-title__select__dropdown-indicator',
    ) as Element;
    await user.click(openButton);

    const loadingSpinner = screen.getByRole('status');
    expect(loadingSpinner).toBeInTheDocument();

    await waitFor(
      () => {
        const loadingSpinner = screen.queryByText(/loading/i);
        expect(loadingSpinner).not.toBeInTheDocument();
      },
      { timeout: 3000 },
    );

    await user.click(screen.getByText('Why Gritty, Phanatic & Swoop Define Philadelphia Sports'));

    await waitFor(async () => {
      const input = screen.getByTestId('title-tag-input');
      expect(input).toHaveValue('Why Gritty, Phanatic & Swoop Define Philadelphia Sports');
    });

    // @todo fix this.
    //expect(dispatchSpy).toHaveBeenCalledWith({
    //  type: 'updateDraft',
    //  draft: { title: 'Why Gritty, Phanatic & Swoop Define Philadelphia Sports' },
    //});
  });

  test('shows brand input field when menu is open', async () => {
    const user = userEvent.setup();
    const { container } = render(
      <AppStateProvider initialState={{ accountId: '12345' }}>
        <CreateTitle />
      </AppStateProvider>,
    );

    const openButton = container.querySelector(
      '.create-title__select__dropdown-indicator',
    ) as Element;
    await user.click(openButton);

    const brandInput = screen.getByPlaceholderText('Enter your brand (optional)');
    expect(brandInput).toBeInTheDocument();
  });

  test('updates brand when typing in brand input', async () => {
    const user = userEvent.setup();
    const { container } = render(
      <AppStateProvider initialState={{ accountId: '12345' }}>
        <CreateTitle />
      </AppStateProvider>,
    );

    const openButton = container.querySelector(
      '.create-title__select__dropdown-indicator',
    ) as Element;
    await user.click(openButton);
    const brandInput = screen.getByPlaceholderText('Enter your brand (optional)');
    await user.type(brandInput, 'My Brand');

    // @todo fix this.
    //expect(dispatchSpy).toHaveBeenCalledWith({
    //  type: 'setBrand',
    //  brand: 'My Brand'
    //});
  });

  test('shows reload button in menu footer', async () => {
    const user = userEvent.setup();
    const { container } = render(
      <AppStateProvider initialState={{ accountId: '12345' }}>
        <CreateTitle />
      </AppStateProvider>,
    );

    const openButton = container.querySelector(
      '.create-title__select__dropdown-indicator',
    ) as Element;
    await user.click(openButton);

    const reloadButton = screen.getByTestId('reload-titles');
    expect(reloadButton).toBeInTheDocument();
  });

  test('calls mutate when reload button is clicked', async () => {
    const user = userEvent.setup();
    const { container } = render(
      <AppStateProvider initialState={{ accountId: '12345' }}>
        <CreateTitle />
      </AppStateProvider>,
    );

    const openButton = container.querySelector(
      '.create-title__select__dropdown-indicator',
    ) as Element;
    await user.click(openButton);

    const reloadButton = screen.getByTestId('reload-titles');
    await user.click(reloadButton);
    // @todo fix this.
    //expect(mockMutate).toHaveBeenCalledWith(undefined, { revalidate: true });
  });

  //test('closes menu when Close button is clicked', async () => {
  //  const user = userEvent.setup();
  //  const { container } = render(
  //    <AppStateProvider initialState={{ accountId: '12345' }}>
  //      <CreateTitle />
  //    </AppStateProvider>,
  //  );

  //  const openButton = container.querySelector('.create-title__select__dropdown-indicator') as Element;
  //  await user.click(openButton);

  //  const loadingSpinner = screen.getByRole('status');
  //  expect(loadingSpinner).toBeInTheDocument();

  //  await waitFor(
  //    () => {
  //      const loadingSpinner = screen.queryByText(/loading/i);
  //      expect(loadingSpinner).not.toBeInTheDocument();
  //    },
  //    { timeout: 3000 },
  //  );

  //  await user.click(screen.getByText('Close'));

  //  await waitFor(() => {
  //    expect(screen.getByRole('listbox')).not.toBeInTheDocument();
  //  });
  //});

  //test('hides text input when menu is open', async () => {
  //  const user = userEvent.setup();
  //  render(<CreateTitle />);

  //  const input = screen.getByRole('textbox', { name: '' });
  //  expect(input).toBeInTheDocument();

  //  await user.click(screen.getByTestId('open-menu'));

  //  expect(screen.queryByRole('textbox', { name: '' })).not.toBeInTheDocument();
  //});

  //test('validates title length correctly', async () => {
  //  const user = userEvent.setup();
  //  render(<CreateTitle />);

  //  const input = screen.getByRole('textbox', { name: '' });
  //  // Type exactly 55 characters (within valid range of 50-61)
  //  await user.type(input, 'A'.repeat(55));

  //  await waitFor(() => {
  //    const indicators = screen.getAllByTestId('form-indicator');
  //    const lengthIndicator = indicators[1]; // Second indicator is for length
  //    expect(lengthIndicator).toHaveAttribute('data-valid', 'true');
  //  });
  //});

  //test('debounces validation keyword mutation', async () => {
  //  vi.useFakeTimers();
  //  vi.mocked(Context.useDraft).mockReturnValue({
  //    id: 'draft-456',
  //    title: 'Test Title',
  //  });

  //  render(<CreateTitle />);

  //  vi.advanceTimersByTime(500);

  //  await waitFor(() => {
  //    expect(mockMutateValidateKeywords).toHaveBeenCalled();
  //  });

  //  vi.useRealTimers();
  //});

  //test('applies open class to form when menu is open', async () => {
  //  const user = userEvent.setup();
  //  const { container } = render(<CreateTitle />);

  //  const form = container.querySelector('.create-title__form');
  //  expect(form).not.toHaveClass('create-title__form--is-open');

  //  await user.click(screen.getByTestId('open-menu'));

  //  expect(form).toHaveClass('create-title__form--is-open');
  //});
});
