import React, { useState, useEffect } from 'react';
import Select, {
  components,
  type DropdownIndicatorProps,
  type ControlProps,
  type GroupBase,
  type SingleValue,
} from 'react-select';
import { useAccountId, useAppStateDispatch, useBrand, useDraft } from '../Context.js';
import { post } from '../utilities/fetcher.js';
import { RotateCw } from 'lucide-react';
import Lightbulb from '../../assets/lightbulb.svg?react';
import Button from '../components/Button.js';
import useSWRImmutable from 'swr/immutable';
import './CreateTitle.css';
import Loading from '../components/Loading.js';
import type { OptionType } from '../utilities/types.js';
import FormIndicator from './FormIndicator.js';

export default function CreateTitle() {
  const dispatch = useAppStateDispatch();
  const accountId = useAccountId();
  const draft = useDraft();
  const brand = useBrand();

  const [menuIsOpen, setMenuIsOpen] = useState(false);
  const [inputValue, setInputValue] = useState('');

  useEffect(() => {
    if (menuIsOpen) {
      mutate(undefined, { revalidate: true });
    }
  }, [menuIsOpen]);

  useEffect(() => {
    if (inputValue !== draft.title && inputValue.length) {
      const timer = setTimeout(() => {
        dispatch({
          type: 'updateDraft',
          draft: {
            title: inputValue,
          },
        });
      }, 500); // Wait 500ms after user stops typing

      return () => {
        clearTimeout(timer);
      };
    }
  }, [inputValue]);

  // @todo this isn't the correct endpoint.
  const { data: dataValidateKeywords, mutate: mutateValidateKeywords } = useSWRImmutable(
    draft.title && draft.id
      ? [
          `/content-guidance-gateway/api/v1/${accountId}/requests/${draft.id}/validate-keywords`,
          {
            title: draft.title,
            metaDescription: '',
            bodyBody: '',
          },
        ]
      : null,
    post,
    { fallbackData: [] },
  );
  // Debounce and call mutate when value stabilises.
  useEffect(() => {
    if (draft.title) {
      const timer = setTimeout(() => {
        mutateValidateKeywords();
      }, 500); // Wait 500ms after user stops typing

      return () => {
        clearTimeout(timer);
      };
    }
  }, [draft.title]);
  const topicValid = !!dataValidateKeywords
    .filter((item: { type: string }) => item.type === 'HTML_TITLE_COMMON')
    .filter((item: { details: boolean }) => Object.values(item.details).includes(true)).length;

  const { data, isLoading, mutate } = useSWRImmutable(
    ['/conductor/proxy/v3/title-tag/content-generation', {}],
    post,
    { fallbackData: { titles: [] } },
  );
  const options =
    data?.titles.map((title: string) => ({
      value: title,
      label: title,
    })) || [];

  const DropdownIndicator = (
    props: DropdownIndicatorProps<OptionType, boolean, GroupBase<OptionType>>,
  ) => {
    return (
      <components.DropdownIndicator {...props}>
        <Lightbulb />
      </components.DropdownIndicator>
    );
  };

  const Control = ({ children, ...props }: ControlProps<OptionType>) => {
    if (menuIsOpen) {
      return;
    }
    return <components.Control {...props}>{children}</components.Control>;
  };

  return (
    <div className="create-title">
      <div className="create-title__header">
        <label className="create-title__label" htmlFor="title-tag">
          Title tag
        </label>
        <div className="create-title__topic">
          <FormIndicator value={inputValue} valid={topicValid} />
          Topic
        </div>
        <div
          className="create-title__length"
          title={
            inputValue.length >= 50 && inputValue.length <= 61
              ? ''
              : 'Title tag falls short of length guidelines.\n' +
                '\n' +
                'Keep your title tag between 50-61 characters.'
          }
        >
          <FormIndicator
            value={inputValue}
            valid={inputValue.length >= 50 && inputValue.length <= 61}
          />
          Length
        </div>
      </div>
      <div className={`create-title__form ${menuIsOpen ? 'create-title__form--is-open' : ''}`}>
        <Select<OptionType, false>
          options={options}
          isLoading={isLoading}
          loadingMessage={() => <Loading />}
          onChange={(selected: SingleValue<OptionType>) =>
            setInputValue(selected ? selected.value : '')
          }
          isSearchable={false}
          components={{ DropdownIndicator, Control }}
          menuIsOpen={menuIsOpen}
          onMenuOpen={() => setMenuIsOpen(true)}
          onMenuClose={() => setMenuIsOpen(false)}
          className="create-title__select"
          classNamePrefix="create-title__select"
        />

        {!menuIsOpen && (
          <input
            data-testid="title-tag-input"
            className="create-title__text-input"
            type="text"
            value={inputValue}
            onChange={(e) => setInputValue(e.target.value)}
            name="title-tag"
            aria-label="Title"
          />
        )}
        {menuIsOpen && (
          <div className="create-title__menu-footer">
            <input
              type="text"
              placeholder="Enter your brand (optional)"
              onChange={(e) =>
                dispatch({
                  type: 'setBrand',
                  brand: e.target.value,
                })
              }
              value={brand}
              className="create-title__brand"
              aria-label="Enter your brand (optional)"
            />
            <button
              data-testid="reload-titles"
              className="create-title__reload"
              disabled={isLoading}
              onClick={() => mutate(undefined, { revalidate: true })}
            >
              <RotateCw />
            </button>
            <Button disabled={isLoading} display="secondary" onClick={() => setMenuIsOpen(false)}>
              Close
            </Button>
          </div>
        )}
      </div>
    </div>
  );
}
