import React from 'react';
import { fireEvent, render, waitFor } from '@testing-library/react';
import '@testing-library/jest-dom';
import { describe, test, expect } from 'vitest';
import { server } from '../mocks/server.js';
import { http, HttpResponse } from 'msw';
import { AppStateProvider } from '../Context.js';
import RankSourceSelector from './RankSourceSelector.js';
import getRankSources200 from '../../../tests/fixtures/api/get-rank-sources-200.json' with { type: 'json' };

describe('RankSourceSelector Component', () => {
  describe('Basic Rendering', () => {
    test('renders fallback option', async () => {
      server.use(
        http.get('/conductor/proxy/v3/:accountId/rank-sources/content-guidance', () => {
          return new HttpResponse(null, { status: 500 });
        }),
      );

      const { container } = render(
        <AppStateProvider initialState={{ accountId: '12345' }}>
          <RankSourceSelector
            defaultRankSource={{
              id: 1,
              description: 'Google (US / English)',
              isoLocale: 'en_US',
            }}
          />
        </AppStateProvider>,
      );

      fireEvent.mouseDown(container.querySelector('.rank-source-selector__dropdown-indicator')!, {
        button: 0,
      });

      await waitFor(() =>
        expect(
          container.querySelectorAll('.rank-source-selector__option[id*="-option-"]'),
        ).toHaveLength(1),
      );

      // Check that the default option exists and has correct text
      const defaultOption = container.querySelector(
        '.rank-source-selector__option[id*="-option-"]',
      );
      expect(defaultOption?.textContent).toBe('Google (US / English)');
    });

    test('renders all options', async () => {
      server.use(
        http.get('/conductor/proxy/v3/:accountId/rank-sources/content-guidance', () => {
          return HttpResponse.json(getRankSources200);
        }),
      );

      const { container } = render(
        <AppStateProvider initialState={{ accountId: '12345' }}>
          <RankSourceSelector
            defaultRankSource={{
              id: 1,
              description: 'Google (US / English)',
              isoLocale: 'en_US',
            }}
          />
        </AppStateProvider>,
      );

      fireEvent.mouseDown(container.querySelector('.rank-source-selector__dropdown-indicator')!, {
        button: 0,
      });

      await waitFor(() => {
        const menu = document.querySelector('.rank-source-selector__menu');
        expect(menu).toBeInTheDocument();
        expect(menu).toBeVisible();
      });

      await waitFor(() =>
        expect(
          container.querySelectorAll('.rank-source-selector__option[id*="-option-"]'),
        ).toHaveLength(223),
      );

      await waitFor(() =>
        expect(
          container.querySelectorAll('.rank-source-selector__option[id*="-option-"]'),
        ).toHaveLength(223),
      );
    });
  });
});
