import React from 'react';
import Select, { type ControlProps, type SingleValue, components } from 'react-select';
import { Globe } from 'lucide-react';
import useSWRImmutable from 'swr/immutable';
import { get } from '../utilities/fetcher.js';
import { useAccountId } from '../Context.js';
import './RankSourceSelector.css';

export interface RankSource {
  id: number;
  description: string;
  isoLocale: string;
}

interface SelectOption {
  value: string;
  label: string;
}

const RankSourceSelector = ({
  defaultRankSource,
  onChange = (_) => {},
}: {
  defaultRankSource: RankSource;
  onChange?: (_: RankSource) => void;
}) => {
  const accountId = useAccountId();
  const { data: rankSourcesData } = useSWRImmutable<RankSource[]>(
    `/conductor/proxy/v3/${accountId}/rank-sources/content-guidance`,
    get,
  );

  const defaultOption: SelectOption = {
    value: `${defaultRankSource.id}`,
    label: defaultRankSource.description,
  };

  const rankSources = rankSourcesData ?? [defaultRankSource];
  const rankSourcesById = rankSources.reduce(
    (acc: { [key: string]: RankSource }, val: RankSource) => {
      const key = `${val.id}`;
      acc[key] = {
        id: val.id,
        description: val.description,
        isoLocale: val.isoLocale,
      };
      return acc;
    },
    {},
  );

  const selectOptions: SelectOption[] = rankSources?.map((rankSource) => ({
    value: `${rankSource.id}`,
    label: rankSource.description,
  }));

  const SelectControl = (props: ControlProps<{ value: string; label: string }, false>) => (
    <components.Control {...props}>
      <Globe size={16} style={{ marginRight: '8px', color: '#6b7280' }} />
      {props.children}
    </components.Control>
  );

  return (
    <Select
      onChange={(newValue: SingleValue<SelectOption>) => {
        const key = newValue?.value;
        if (key && rankSourcesById[key]) {
          onChange(rankSourcesById[key]);
        }
      }}
      options={selectOptions}
      components={{
        Control: SelectControl,
      }}
      defaultValue={defaultOption}
      className="rank-source-selector__container"
      classNamePrefix="rank-source-selector"
      unstyled
      isSearchable={true}
      isMulti={false}
      aria-label="Choose a rank source option"
    />
  );
};

export default RankSourceSelector;
