import { render, screen, fireEvent } from '@testing-library/react';
import { describe, test, expect, vi } from 'vitest';
import { userEvent } from '@testing-library/user-event';
import TagInput from './TagInput.js';

describe('TagInput', () => {
  test('renders with default props', () => {
    render(<TagInput />);

    const container = screen.getByRole('textbox');
    expect(container).toBeInTheDocument();
    expect(container).toHaveClass('tag-input__input');
  });

  test('renders with custom placeholder', () => {
    render(<TagInput placeholder="Enter tags..." />);

    expect(screen.getByPlaceholderText('Enter tags...')).toBeInTheDocument();
  });

  test('renders with custom aria-label', () => {
    render(<TagInput aria-label="Tag input field" />);

    expect(screen.getByLabelText('Tag input field')).toBeInTheDocument();
  });

  test('applies correct CSS classes', () => {
    render(<TagInput />);

    expect(document.querySelector('.tag-input__container')).toBeInTheDocument();
    expect(document.querySelector('.tag-input__tags')).toBeInTheDocument();
  });

  describe('Adding tags', () => {
    test('adds tag when Enter is pressed', () => {
      render(<TagInput />);

      const input = screen.getByRole('textbox');
      fireEvent.change(input, { target: { value: 'test-tag' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      expect(screen.getByText('test-tag')).toBeInTheDocument();
      expect(input).toHaveValue('');
    });

    test('adds tag when comma is pressed', () => {
      render(<TagInput />);

      const input = screen.getByRole('textbox');
      fireEvent.change(input, { target: { value: 'test-tag' } });
      fireEvent.keyDown(input, { key: ',', code: 'Comma' });

      expect(screen.getByText('test-tag')).toBeInTheDocument();
      expect(input).toHaveValue('');
    });

    test('adds tag on blur', () => {
      render(<TagInput />);

      const input = screen.getByRole('textbox');
      fireEvent.change(input, { target: { value: 'test-tag' } });
      fireEvent.blur(input);

      expect(screen.getByText('test-tag')).toBeInTheDocument();
    });

    test('trims whitespace from tags', () => {
      render(<TagInput />);

      const input = screen.getByRole('textbox');
      fireEvent.change(input, { target: { value: '  test-tag  ' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      expect(screen.getByText('test-tag')).toBeInTheDocument();
    });

    test('does not add empty tags', () => {
      render(<TagInput />);

      const input = screen.getByRole('textbox');
      fireEvent.change(input, { target: { value: '   ' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      expect(screen.queryByText('   ')).not.toBeInTheDocument();
    });

    test('does not add duplicate tags', () => {
      render(<TagInput />);

      const input = screen.getByRole('textbox');

      // Add first tag
      fireEvent.change(input, { target: { value: 'test-tag' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      // Try to add same tag again
      fireEvent.change(input, { target: { value: 'test-tag' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      const tags = screen.getAllByText('test-tag');
      expect(tags).toHaveLength(1);
    });

    test('respects maxTags limit', () => {
      render(<TagInput maxTags={2} />);

      const input = screen.getByRole('textbox');

      // Add first tag
      fireEvent.change(input, { target: { value: 'tag1' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      // Add second tag
      fireEvent.change(input, { target: { value: 'tag2' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      // Try to add third tag
      fireEvent.change(input, { target: { value: 'tag3' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      expect(screen.getByText('tag1')).toBeInTheDocument();
      expect(screen.getByText('tag2')).toBeInTheDocument();
      expect(screen.queryByText('tag3')).not.toBeInTheDocument();
    });

    test('hides input when maxTags limit is reached', () => {
      render(<TagInput maxTags={1} />);

      const input = screen.getByRole('textbox');
      fireEvent.change(input, { target: { value: 'tag1' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      expect(screen.queryByRole('textbox')).not.toBeInTheDocument();
    });
  });

  describe('Removing tags', () => {
    test('removes tag when remove button is clicked', () => {
      render(<TagInput />);

      const input = screen.getByRole('textbox');
      fireEvent.change(input, { target: { value: 'test-tag' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      const removeButton = screen.getByLabelText('Remove test-tag');
      fireEvent.click(removeButton);

      expect(screen.queryByText('test-tag')).not.toBeInTheDocument();
    });

    test('removes last tag when Backspace is pressed on empty input', () => {
      render(<TagInput />);

      const input = screen.getByRole('textbox');

      // Add two tags
      fireEvent.change(input, { target: { value: 'tag1' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });
      fireEvent.change(input, { target: { value: 'tag2' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      // Press backspace on empty input
      fireEvent.keyDown(input, { key: 'Backspace', code: 'Backspace' });

      expect(screen.getByText('tag1')).toBeInTheDocument();
      expect(screen.queryByText('tag2')).not.toBeInTheDocument();
    });

    test('does not remove tag when Backspace is pressed on non-empty input', () => {
      render(<TagInput />);

      const input = screen.getByRole('textbox');

      // Add a tag
      fireEvent.change(input, { target: { value: 'tag1' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      // Type something in input and press backspace
      fireEvent.change(input, { target: { value: 'text' } });
      fireEvent.keyDown(input, { key: 'Backspace', code: 'Backspace' });

      expect(screen.getByText('tag1')).toBeInTheDocument();
    });

    test('shows input again after removing tags when below maxTags', () => {
      render(<TagInput maxTags={1} />);

      // Add tag to reach limit
      const input = screen.getByRole('textbox');
      fireEvent.change(input, { target: { value: 'tag1' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      // Input should be hidden
      expect(screen.queryByRole('textbox')).not.toBeInTheDocument();

      // Remove tag
      const removeButton = screen.getByLabelText('Remove tag1');
      fireEvent.click(removeButton);

      // Input should be visible again
      expect(screen.getByRole('textbox')).toBeInTheDocument();
    });
  });

  describe('Placeholder behavior', () => {
    test('shows placeholder only when no tags exist', () => {
      render(<TagInput placeholder="Enter tags..." />);

      const input = screen.getByRole('textbox');
      expect(input).toHaveAttribute('placeholder', 'Enter tags...');

      // Add a tag
      fireEvent.change(input, { target: { value: 'tag1' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      // Placeholder should be empty now
      const inputAfterTag = screen.getByRole('textbox');
      expect(inputAfterTag).toHaveAttribute('placeholder', '');
    });

    test('does not show placeholder when tags exist', () => {
      render(<TagInput placeholder="Enter tags..." />);

      const input = screen.getByRole('textbox');

      // Add a tag
      fireEvent.change(input, { target: { value: 'existing-tag' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      // Verify tag exists and placeholder is not present
      expect(screen.getByText('existing-tag')).toBeInTheDocument();
      expect(input).toHaveAttribute('placeholder', '');
      expect(input).not.toHaveAttribute('placeholder', 'Enter tags...');
    });
  });

  describe('Container interaction', () => {
    test('focuses input when container is clicked', () => {
      render(<TagInput />);

      const container = document.querySelector('.tag-input__container') as Element;
      const input = screen.getByRole('textbox');

      fireEvent.click(container);

      expect(input).toHaveFocus();
    });
  });

  describe('onValueChange callback', () => {
    test('calls onValueChange when tags are added', () => {
      const mockOnValueChange = vi.fn();
      render(<TagInput onValueChange={mockOnValueChange} />);

      const input = screen.getByRole('textbox');
      fireEvent.change(input, { target: { value: 'test-tag' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      // Should be called twice: once on mount (empty array), once when tag is added
      expect(mockOnValueChange).toHaveBeenCalledTimes(2);
      expect(mockOnValueChange).toHaveBeenNthCalledWith(1, []);
      expect(mockOnValueChange).toHaveBeenNthCalledWith(2, ['test-tag']);
    });

    test('calls onValueChange when tags are removed', () => {
      const mockOnValueChange = vi.fn();
      render(<TagInput onValueChange={mockOnValueChange} />);

      const input = screen.getByRole('textbox');

      // Add tag
      fireEvent.change(input, { target: { value: 'test-tag' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      // Remove tag
      const removeButton = screen.getByLabelText('Remove test-tag');
      fireEvent.click(removeButton);

      // Should be called: mount, add, remove
      expect(mockOnValueChange).toHaveBeenCalledTimes(3);
      expect(mockOnValueChange).toHaveBeenNthCalledWith(3, []);
    });

    test('calls onValueChange with correct tag order', () => {
      const mockOnValueChange = vi.fn();
      render(<TagInput onValueChange={mockOnValueChange} />);

      const input = screen.getByRole('textbox');

      // Add multiple tags
      fireEvent.change(input, { target: { value: 'tag1' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });
      fireEvent.change(input, { target: { value: 'tag2' } });
      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      expect(mockOnValueChange).toHaveBeenLastCalledWith(['tag1', 'tag2']);
    });
  });

  describe('Input value management', () => {
    test('clears input value after adding tag', () => {
      render(<TagInput />);

      const input = screen.getByRole('textbox');
      fireEvent.change(input, { target: { value: 'test-tag' } });

      expect(input).toHaveValue('test-tag');

      fireEvent.keyDown(input, { key: 'Enter', code: 'Enter' });

      expect(input).toHaveValue('');
    });

    test('updates input value on change', () => {
      render(<TagInput />);

      const input = screen.getByRole('textbox');
      fireEvent.change(input, { target: { value: 'typing...' } });

      expect(input).toHaveValue('typing...');
    });
  });

  describe('Edge cases', () => {
    test('handles multiple rapid tag additions', async () => {
      const user = userEvent.setup();
      render(<TagInput />);

      const input = screen.getByRole('textbox');

      // Add multiple tags quickly
      for (const tagName of ['tag1', 'tag2', 'tag3']) {
        await user.type(input, tagName);
        await user.keyboard('{Enter}');
      }

      expect(screen.getByText('tag1')).toBeInTheDocument();
      expect(screen.getByText('tag2')).toBeInTheDocument();
      expect(screen.getByText('tag3')).toBeInTheDocument();
    });

    test('handles removing non-existent tag index gracefully', async () => {
      const user = userEvent.setup();
      render(<TagInput />);

      const input = screen.getByRole('textbox');
      await user.type(input, 'tag1');
      await user.keyboard('{Enter}');

      // This shouldn't break the component
      expect(async () => {
        // Simulate clicking remove on a tag that doesn't exist
        const removeButton = screen.getByLabelText('Remove tag1');
        await user.click(removeButton);
        // The button won't exist anymore, but the test should still pass
      }).not.toThrow();
    });
  });
});
