import React, { useEffect, useState } from 'react';
import Tag from './Tag.js';
import './TagInput.css';

function TagInput({
  maxTags = 5,
  onValueChange = () => {},
  placeholder = '',
  initialKeywords = [],
  ...props
}: {
  maxTags?: number;
  onValueChange?: (tags: string[]) => void;
  placeholder?: string;
  initialKeywords?: string[];
}) {
  const [tags, setTags] = useState<string[]>(initialKeywords);
  const [currentInput, setCurrentInput] = useState<string>('');
  const tagCount: number = tags.length;

  useEffect(() => {
    onValueChange(tags);
  }, [tags]);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement>): void => {
    setCurrentInput(e.target.value);
  };

  const handleInputKeyDown = (e: React.KeyboardEvent<HTMLInputElement>): void => {
    if (e.key === 'Enter' || e.key === ',') {
      e.preventDefault();
      addTag();
    } else if (e.key === 'Backspace' && currentInput === '' && tags.length > 0) {
      removeTag(tags.length - 1);
    }
  };

  const handleTagRemoveClick = (e: React.MouseEvent<HTMLButtonElement>, index: number): void => {
    e.stopPropagation();
    removeTag(index);
  };

  const addTag = (): void => {
    const trimmed = currentInput.trim();
    if (trimmed && tagCount < maxTags && !tags.includes(trimmed)) {
      setTags([...tags, trimmed]);
      setCurrentInput('');
    }
  };

  const removeTag = (index: number): void => {
    setTags(tags.filter((_, i) => i !== index));
  };

  return (
    <div
      className="tag-input__container"
      onClick={() => {
        document.getElementById('conductor__tag-input')?.focus();
      }}
    >
      <div className="tag-input__tags">
        {tags.map((tag, index) => (
          <Tag key={tag} remove={(e) => handleTagRemoveClick(e, index)}>
            {tag}
          </Tag>
        ))}
        {tagCount < maxTags && (
          <input
            id="conductor__tag-input"
            type="text"
            value={currentInput}
            onChange={handleInputChange}
            onKeyDown={handleInputKeyDown}
            onBlur={addTag}
            placeholder={tagCount === 0 ? placeholder : ''}
            className="tag-input__input"
            {...props}
          />
        )}
      </div>
    </div>
  );
}

export default TagInput;
