<?php

declare(strict_types=1);

namespace Drupal\conductor;

use Drupal\conductor\Exception\ConductorCredentialsNotFoundException;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\key\KeyRepositoryInterface;
use GuzzleHttp\Client;
use Psr\Log\LoggerInterface;
use Symfony\Bridge\PsrHttpMessage\Factory\HttpFoundationFactory;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Defines an HTTP API client for interacting with the Conductor API.
 */
final class ConductorHttpApiClient {

  // @todo improve naming. Temporary
  public const string CACHE_KEY = 'conductor_api';

  public function __construct(
    private readonly Client $client,
    private readonly HttpFoundationFactory $httpFoundationFactory,
    private readonly ConfigFactory $configFactory,
    private readonly LoggerInterface $logger,
    private readonly AccountProxyInterface $currentUser,
    private readonly CacheBackendInterface $cache,
    private readonly KeyRepositoryInterface $keyRepository,
    private readonly string $baseUrl,
  ) {}

  /**
   * Forward the request to Conductor API and returns its response.
   *
   * @param string $resource
   *   The uri to forward to Conductor API.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object from Drupal.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The response from Conductor API.
   */
  public function forward(string $resource, Request $request): JsonResponse {
    // Method.
    $method = $request->getMethod();

    // Options (headers, body,...)
    $options = $this->buildRequestOptions($request);

    try {
      $response = $this->request($method, $resource, $options, $request);
      $contentType = (string) $response->headers->get('Content-Type', '');
      $normalizedContentType = strtolower(trim(explode(';', $contentType)[0]));

      return match($normalizedContentType) {
        'application/json' => $this->handleJsonResponse($response),
        'text/html' => $this->handleHtmlResponse($response),
        default => new JsonResponse([
          'error' => 'Invalid response',
          'content' => (string) $response->getContent(),
        ], $response->getStatusCode(), ['Content-Type' => 'application/json']),
      };

    }
    catch (ConductorCredentialsNotFoundException $e) {
      $this->logger->error($e->getMessage());
      return new JsonResponse(['error' => $e->getMessage()], Response::HTTP_UNAUTHORIZED);
    }
    catch (\Exception $e) {
      $this->logger->error($e->getMessage());
      return new JsonResponse(['error' => $e->getMessage()], Response::HTTP_INTERNAL_SERVER_ERROR);
    }
  }

  private function handleJsonResponse(Response $response): JsonResponse {
    $body = json_decode((string) $response->getContent(), associative: TRUE, flags: JSON_THROW_ON_ERROR);
    return new JsonResponse($body, $response->getStatusCode(), $response->headers->all());
  }

  private function handleHtmlResponse(Response $response): JsonResponse {
    // Set the content type to json and reuse the same headers for the JsonResponse.
    $response->headers->set('Content-Type', 'application/json');
    return new JsonResponse(
      ['content' => (string) $response->getContent()],
      $response->getStatusCode(),
      $response->headers->all(),
    );
  }

  /**
   * @param string $method
   *   The HTTP method.
   * @param string $uri
   *   The URI.
   * @param array<mixed> $options
   *   The options.
   * @param \Symfony\Component\HttpFoundation\Request|null $forwardedRequest
   *   The forwarded request to Conductor, if any.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response from Conductor API.
   * @throws \GuzzleHttp\Exception\GuzzleException
   * @throws \Drupal\conductor\Exception\ConductorCredentialsNotFoundException
   */
  public function request(string $method, string $uri, array $options = [], ?Request $forwardedRequest = NULL): Response {
    $settings = $this->configFactory->get('conductor.settings');
    $uri = $this->buildRequestUrl($uri, $forwardedRequest);

    $psr7Response = $this->client->request($method, $uri, $options);
    $response = $this->httpFoundationFactory->createResponse($psr7Response);
    if ($settings->get('verbose_logging')) {
      $this->logger->info("Request to Conductor API<br> <ul>" .
          "<li><strong>Method</strong>: @method</li>" .
          "<li><strong>Conductor URL</strong>: @uri</li>" .
          "<li><strong>Drupal Request</strong>: @request</li>" .
          "<li><strong>Conductor Response</strong>: @response</li></ul>",
         [
           '@method' => $method,
           '@uri' => $uri,
           '@request' => $forwardedRequest ?? new Request(),
           '@response' => $response,
         ]
      );
    }
    return $response;
  }

  /**
   * Obtains from the conductor API the account id.
   *
   * @return string
   *   The account id from Conductor API.
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   * @throws \Drupal\conductor\Exception\ConductorCredentialsNotFoundException
   */
  public function getAccountId(): ?string {
    $cid = \sprintf('%s:%s:user:%s', self::CACHE_KEY, 'account_id', $this->currentUser->id());
    $cached = $this->cache->get($cid);
    if ($cached) {
      \assert(!empty($cached->data));
      \assert(\is_string($cached->data));
      return $cached->data;
    }

    $response = $this->request('GET', 'v3/accounts');
    $accounts = json_decode((string) $response->getContent(), TRUE);

    $this->cache->set($cid, $accounts[0]['accountId'], tags: [
      self::CACHE_KEY . ":account_id",
      "user:" . $this->currentUser->id(),
    ]);

    return $accounts[0]['accountId'];
  }

  /**
   * Gets the API credentials.
   *
   * @return array
   *   Array containing 'api_key' and 'shared_secret', or empty array if not available.
   */
  private function getCredentials(): array {
    $config = $this->configFactory->get('conductor.settings');
    $keyId = $config->get('key_id');
    if (NULL === $keyId) {
      return [];
    }

    $key = $this->keyRepository->getKey($keyId);
    if (NULL === $key) {
      return [];
    }

    try {
      $credentials = json_decode($key->getKeyValue(), TRUE, flags: JSON_THROW_ON_ERROR);
      return [
        'api_key' => $credentials['api_key'],
        'shared_secret' => $credentials['shared_secret'],
      ];
    }
    catch (\Exception) {
      return [];
    }
  }

  /**
   * Builds the options for the Guzzle request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The Request object.
   *
   * @return array<mixed>
   *   The options for the Guzzle request.
   */
  private function buildRequestOptions(Request $request): array {
    // @todo should we use the request headers?
    // Transfer headers could be a security risk
    // @see https://medium.com/@0xwan/hop-by-hop-header-78d0866101f6
    $headers = [
      'Accept' => 'application/json',
    ];

    $body = '';
    if (!in_array($request->getMethod(), ['GET', 'HEAD'], TRUE)) {
      $headers['Content-Type'] = 'application/json';
      $body = $request->getContent(TRUE) ?? '';
    }

    return [
      'headers' => $headers,
      'body' => $body,
      // And the default options.
      'timeout' => 30,
      'connect_timeout' => 30,
      'http_errors' => FALSE,
    ];
  }

  /**
   * Build the URL string for the Conductor API request.
   *
   * @param string $uri
   *   The Conductor API resource to fetch.
   * @param \Symfony\Component\HttpFoundation\Request|null $request
   *   The Conductor API resource to fetch.
   *
   * @return string
   *   The URL string with the authentication.
   *
   * @throws \Drupal\conductor\Exception\ConductorCredentialsNotFoundException
   */
  private function buildRequestUrl(string $uri, ?Request $request = NULL): string {
    $normalizedPath = ltrim($uri, '/');
    $base = rtrim($this->baseUrl, '/') . '/' . $normalizedPath;
    $query_string = '';
    if ($request) {
      $query_string = $request->getQueryString();
    }

    $credentials = $this->getCredentials();
    if (empty($credentials['api_key']) || empty($credentials['shared_secret'])) {
      throw new ConductorCredentialsNotFoundException('Unable to connect to Conductor API. The API credentials are not configured or not available.');
    }

    $query = http_build_query([
      'apiKey' => $credentials['api_key'],
      'sig' => $this->constructSignature($credentials['api_key'], $credentials['shared_secret']),
    ], '', '&', PHP_QUERY_RFC3986) .
      '&' . $query_string;
    return "$base?$query";
  }

  /**
   * Construct signature for API requests to Conductor.
   *
   * @param string|null $api_key
   *   The API key to use. If NULL, will get from credentials.
   * @param string|null $shared_secret
   *   The shared secret to use. If NULL, will get from credentials.
   *
   * @return string
   *   A signature for API requests to Conductor.
   */
  private function constructSignature(?string $api_key = NULL, ?string $shared_secret = NULL): string {
    if ($api_key === NULL || $shared_secret === NULL) {
      $credentials = $this->getCredentials();
      $api_key = $credentials['api_key'] ?? '';
      $shared_secret = $credentials['shared_secret'] ?? '';
    }

    if (empty($api_key) || empty($shared_secret)) {
      throw new \RuntimeException('API credentials not available for signature construction.');
    }

    return md5($api_key . $shared_secret . time());
  }

}
