<?php

declare(strict_types=1);

namespace Drupal\conductor\Repository;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\conductor\Exception\UnableToAssignDraftToEntityException;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

final readonly class DraftRepository implements DraftRepositoryInterface {

  public const string TABLE_NAME = 'conductor_draft_map';

  public function __construct(
    private Connection $connection,
    private TimeInterface $time,
    private EntityTypeManagerInterface $entityTypeManager,
  ) {}

  public function getDrafts(): array {
    $result = $this->connection->select(self::TABLE_NAME)
      ->fields(self::TABLE_NAME)
      ->execute();
    if (!$result instanceof StatementInterface) {
      return [];
    }
    return $result->fetchAllAssoc('entity_id', \PDO::FETCH_ASSOC);
  }

  public function getDraft(string $entity_type, int $entity_id): array {
    $result = $this->connection->select(self::TABLE_NAME)
      ->fields(self::TABLE_NAME)
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->execute();
    if (!$result instanceof StatementInterface) {
      return [];
    }
    $draft = $result->fetchAssoc();
    if (!is_array($draft) || empty($draft)) {
      return [];
    }
    return $draft;
  }

  /**
   * @throws \Drupal\conductor\Exception\UnableToAssignDraftToEntityException
   */
  public function upsertDraft(string $entity_type, int $entity_id, string $draft_id): void {
    $entity = $this->entityTypeManager->getStorage($entity_type)->load($entity_id);
    if (NULL === $entity) {
      throw new UnableToAssignDraftToEntityException("$entity_type $entity_id not found");
    }

    $time = $this->time->getCurrentTime();
    $this->connection->merge(self::TABLE_NAME)
      ->keys([
        'entity_id' => $entity_id,
        'entity_type' => $entity_type,
      ])
      ->insertFields([
        'entity_id' => $entity_id,
        'entity_type' => $entity_type,
        'draft_id' => $draft_id,
        'created' => $time,
        'changed' => $time,
      ])
      ->updateFields([
        'draft_id' => $draft_id,
        'changed' => $time,
      ])
      ->execute();
  }

  /**
   * Insert a new draft-entity_id association into the database.
   *
   * @throws \Drupal\conductor\Exception\UnableToAssignDraftToEntityException
   */
  public function createDraft(string $entity_type, int $entity_id, string $draft_id): void {
    if (
      $this->isEntityAssociatedWithDraft($entity_type, $entity_id) ||
      $this->isDraftAssociatedWithEntity($draft_id)
    ) {
      throw new UnableToAssignDraftToEntityException("Draft $draft_id already associated with another canvas page.");
    }

    $this->upsertDraft($entity_type, $entity_id, $draft_id);
  }

  /**
   * @throws \Drupal\conductor\Exception\UnableToAssignDraftToEntityException
   */
  public function updateDraft(string $entity_type, int $entity_id, string $draft_id): void {
    $this->upsertDraft($entity_type, $entity_id, $draft_id);
  }

  public function deleteDraft(string $entity_type, int $entity_id): bool {
    return (bool) $this->connection->update(self::TABLE_NAME)
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->fields(['deleted' => 1])
      ->execute();
  }

  private function isEntityAssociatedWithDraft(string $entity_type, int $entity_id): bool {
    $result = $this->connection->select(self::TABLE_NAME)
      ->fields(self::TABLE_NAME)
      ->condition('entity_type', $entity_type)
      ->condition('entity_id', $entity_id)
      ->execute();
    if (!$result instanceof StatementInterface) {
      return FALSE;
    }
    return (bool) $result->fetchField();
  }

  private function isDraftAssociatedWithEntity(string $draft_id): bool {
    $result = $this->connection->select(self::TABLE_NAME)
      ->fields(self::TABLE_NAME)
      ->condition('draft_id', $draft_id)
      ->execute();
    if (!$result instanceof StatementInterface) {
      return FALSE;
    }
    return (bool) $result->fetchField();
  }

}
