<?php

declare(strict_types=1);

namespace Drupal\conductor_test\Plugin\ServiceMock;

use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\http_request_mock\Attribute\ServiceMock;
use Drupal\http_request_mock\ServiceMockPluginInterface;
use GuzzleHttp\Psr7\Response;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Intercepts any HTTP request made to app or api.conductor.com.
 */
#[ServiceMock(
  id: 'conductor_com',
  label: new TranslatableMarkup('conductor.com'),
  weight: 0,
)]

final class ApiConductorComPlugin extends PluginBase implements ServiceMockPluginInterface, ContainerFactoryPluginInterface {

  /**
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  private ModuleExtensionList $moduleExtensionList;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, ModuleExtensionList $module_extension_list) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->moduleExtensionList = $module_extension_list;

  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new self(
      $configuration, $plugin_id, $plugin_definition,
      $container->get('extension.list.module'),
    );
  }

  private const array DEFAULT_JSON_HEADER = [
    'Content-Type' => 'application/json; charset=UTF-8; version=1.0; profile="https://example.com/schema/profile"; action="create"; x-custom-param="some-value"',
    'Cache-Control' => 'max-age=0, must-revalidate, no-cache, no-store, private',
    'connection' => 'keep-alive',
    'vary' => 'Accept-Encoding, Origin, Access-Control-Request-Method, Access-Control-Request-Headers',
  ];
  private const array WRONG_JSON_HEADER = [
    'Content-Type' => 'charset=UTF-8; application/json;  version=1.0; profile="https://example.com/schema/profile"; action="create"; x-custom-param="some-value"',
    'Cache-Control' => 'max-age=0, must-revalidate, no-cache, no-store, private',
    'connection' => 'keep-alive',
    'vary' => 'Accept-Encoding, Origin, Access-Control-Request-Method, Access-Control-Request-Headers',
  ];
  private const array DEFAULT_HTML_HEADER = [
    'Content-Type' => 'text/html; charset=UTF-8',
    'Cache-Control' => 'max-age=0, must-revalidate, no-cache, no-store, private',
    'connection' => 'keep-alive',
    'vary' => 'Accept-Encoding, Origin, Access-Control-Request-Method, Access-Control-Request-Headers',
  ];

  /**
   * {@inheritdoc}
   */
  public function applies(RequestInterface $request, array $options): bool {
    return $request->getUri()->getHost() === 'api.conductor.com' || $request->getUri()->getHost() === 'api.conductor.com' || $request->getUri()->getHost() === 'app.conductor.com';
  }

  /**
   * {@inheritdoc}
   */
  public function getResponse(RequestInterface $request, array $options): ResponseInterface {
    return match($request->getMethod()){
      'GET' => $this->handleGet($request, $options),
      'POST' => $this->handlePost($request, $options),
      'PUT' => $this->handlePut($request, $options),
      'DELETE' => $this->handleDelete($request, $options),
      default => new Response(200, [], 'Mocking example.com response'),
    };
  }

  private function handleGet(RequestInterface $request, array $options): ResponseInterface {
    $path = $request->getUri()->getPath();

    if (preg_match('/^\/v3\/(\w+)\/rank-sources\/content-guidance$/', $path, $matches)) {
      return $this->getJson('get-rank-sources-200.json');
    }

    if (preg_match('/^\/v3\/accounts\/(\w+)\/batch-untracked-serp\/serp-explorer$/', $path, $matches)) {
      return $this->getJson('get-batch-untracked-serp-200.json');
    }

    if (preg_match('/^\/v3\/(\w+)\/content-guidance$/', $path, $matches)) {
      return $this->getJson('get-content-guidance-200.json');
    }

    return match($path) {
      '/v3/accounts' => $this->accounts200(),
      '/v3/accounts-500' => $this->accounts500(),
      '/v3/wrong-json-header' => $this->wrongJsonHeader(),
      default => new Response(404, self::DEFAULT_HTML_HEADER, '<h1>Not found</h1>'),
    };
  }

  private function handlePost(RequestInterface $request, array $options): ResponseInterface {
    $path = $request->getUri()->getPath();

    if (preg_match('/^\/v3\/(\w+)\/content-guidance$/', $path, $matches)) {
      return $this->getJson('post-content-guidance-200.json');
    }

    return new Response(404, self::DEFAULT_HTML_HEADER, '<h1>Not found</h1>');
  }

  private function handlePut(RequestInterface $request, array $options): ResponseInterface {
    return new Response(200, [], 'Mocking example.com response');
  }

  private function handleDelete(RequestInterface $request, array $options): ResponseInterface {
    return new Response(200, [], 'Mocking example.com response');
  }

  private function accounts200():Response {
    return new Response(
      200,
      self::DEFAULT_JSON_HEADER,
      '[{"accountId":"12345","name":"Account Name","isActive":true,"webProperties":"https://api.conductor.com/v3/accounts/12345/web-properties"}]'
    );
  }

  private function accounts500():Response {
    return new Response(
      500,
      self::DEFAULT_HTML_HEADER,
      '<h1>Internal Server Error</h1>'
    );
  }

  private function wrongJsonHeader():Response {
    return new Response(
      200,
      self::WRONG_JSON_HEADER,
      '[{"accountId":"24936","name":"Product Dev_Acquia (Promet Source)","isActive":true,"webProperties":"https://api.conductor.com/v3/accounts/24936/web-properties"}]'
    );
  }

  private function getJson(string $filename): ResponseInterface {
    $module_path = $this->moduleExtensionList->getPath('conductor');
    $json_file = $module_path . "/tests/fixtures/api/$filename";

    if (file_exists($json_file)) {
      $json_content = file_get_contents($json_file);
      if (is_string($json_content)) {
        return new Response(200, ['Content-Type' => 'application/json'], $json_content);
      }
    }

    return new Response(500, self::DEFAULT_HTML_HEADER, '<h1>Internal Server Error</h1>');
  }

}
