<?php

declare(strict_types=1);

namespace Drupal\Tests\conductor\Functional;

use Drupal\Core\Session\AccountInterface;
use PHPUnit\Framework\Attributes\Group;
use Symfony\Component\HttpFoundation\Response;

/**
 * Browser tests for conductor module.
 */
#[Group("conductor")]
final class ConductorHttpApiBrowserTest extends ConductorBrowserTestBase {

  /**
   * {@inheritdoc}
   *
   * Enable only the module under test for now.
   */
  protected static $modules = [
    'conductor',
    'conductor_test',
    'key',
  ];

  /**
   * {@inheritdoc}
   *
   * Use a minimal core theme to avoid unrelated dependencies.
   */
  protected $defaultTheme = 'stark';

  /**
   * Request the simplest GET request to the Conductor API using the HTTP client.
   */
  public function testSimpleGetRequests(): void {
    $this->generateKey();
    // Anonymous user.
    $this->drupalGet('/conductor/proxy/v3/accounts');
    $this->assertSession()->statusCodeEquals(403);
    $this->assertSession()->responseContains('You are not authorized to access this page');

    // Logged in user with permissions.
    $account = $this->drupalCreateUser(['use conductor', 'access content']);
    assert($account instanceof AccountInterface);
    $this->drupalLogin($account);
    $this->assertFalse(\Drupal::currentUser()->isAnonymous(), 'User should not be anonymous after login');
    $this->assertTrue($this->drupalUserIsLoggedIn($account));
    $this->drupalGet('/conductor/proxy/v3/accounts');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseContains('{"accountId":"');
    // non-existing endpoint.
    $this->drupalGet('/conductor/proxy/v3/does-not-exist');
    $this->assertSession()->statusCodeEquals(404);
    $this->assertSession()->responseContains('Not found');

    // Internal server error.
    $this->drupalGet('/conductor/proxy/v3/accounts-500');
    $this->assertSession()->statusCodeEquals(500);
    $this->assertSession()->responseContains('Internal Server Error');

    // Invalid json header.
    $this->drupalGet('/conductor/proxy/v3/wrong-json-header');
    $this->assertSession()->statusCodeEquals(200);
    // There is an invalid response
    $this->assertSession()->responseContains('Invalid response');
    // But in the content, there is the json.
    $this->assertSession()->responseContains('accountId');
  }

  public function testRequestsWithoutCredentials(): void {
    // Anonymous user.
    $this->drupalGet('/conductor/proxy/v3/accounts');
    $this->assertSession()->statusCodeEquals(403);
    $this->assertSession()->responseContains('You are not authorized to access this page');

    // Logged in user with permissions.
    $account = $this->drupalCreateUser(['use conductor', 'access content']);
    assert($account instanceof AccountInterface);
    $this->drupalLogin($account);
    $this->assertFalse(\Drupal::currentUser()->isAnonymous(), 'User should not be anonymous after login');
    $this->assertTrue($this->drupalUserIsLoggedIn($account));
    $this->drupalGet('/conductor/proxy/v3/accounts');
    $this->assertSession()->statusCodeEquals(Response::HTTP_UNAUTHORIZED);
    $this->assertSession()->responseContains('Unable to connect to Conductor API. The API credentials are not configured or not available');
  }

}
