<?php

declare(strict_types=1);

namespace Drupal\Tests\conductor\FunctionalJavascript;

use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\canvas\Entity\Component;
use Drupal\canvas\Entity\Page;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\conductor\Trait\KeyTrait;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests basic JavaScript functionality.
 */
#[Group("conductor")]
class DrupalSettingsFunctionalTest extends WebDriverTestBase {

  use KeyTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Modules to enable.
   *
   * @var string[]
   */
  protected static $modules = [
    'node',
    'user',
    'canvas',
    'conductor',
    'conductor_test',
    'key',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->generateKey();
  }

  /**
   * Tests drupalSettings values are correct.
   */
  public function testAccountIdInDrupalSettings(): void {
    // Anonymous users should not have conductor settings.
    $this->drupalGet('<front>');
    $drupalSettings = $this->getDrupalSettings();
    $this->assertArrayNotHasKey('conductor', $drupalSettings);

    $editor = $this->createUser([
      'use conductor',
      'access content',
      Component::ADMIN_PERMISSION,
      Page::CREATE_PERMISSION,
      Page::EDIT_PERMISSION,
    ], admin: TRUE);
    $this->assertInstanceOf(AccountInterface::class, $editor);
    $this->drupalLogin($editor);
    $this->drupalGet('<front>');
    $drupalSettings = $this->getDrupalSettings();
    // drupalSettings.conductor should only be present in the Canvas editor.
    $this->assertArrayNotHasKey('conductor', $drupalSettings);
    $this->drupalLogout();

    // Logged-in user with permissions and in the Canvas Editor should have
    // conductor settings present
    $page = Page::create([
      'title' => "Page 1",
      'status' => TRUE,
      'path' => ['alias' => "/page-1"],
    ]);
    $page->save();
    \assert($page instanceof Page);
    $this->drupalLogin($editor);
    $url = Url::fromRoute('canvas.boot.entity', ['entity_type' => 'canvas_page', 'entity' => $page->id()]);
    $this->drupalGet($url);
    $drupalSettings = $this->getDrupalSettings();
    $this->assertArrayHasKey('conductor', $drupalSettings);
    $this->assertArrayHasKey('accountId', $drupalSettings['conductor']);
    $this->assertEquals('12345', $drupalSettings['conductor']['accountId']);
    $this->assertArrayNotHasKey('draftId', $drupalSettings['conductor']);

    // DraftId should only be present if there is a draft.
    // \Drupal::service(DraftRepository::class)->createDraft((int) $page->id(), 'the-uuid-for-a-draft-id');
    // $this->drupalGet($url);
    // $drupalSettings = $this->getDrupalSettings();
    // $this->assertArrayHasKey('draftId', $drupalSettings['conductor']);
    // $this->assertEquals('the-uuid-for-a-draft-id', $drupalSettings['conductor']['draftId']);
  }

}
