<?php

namespace Drupal\config2php\Form;

use Drupal\config\Form\ConfigSingleExportForm;
use Drupal\config2php\Service\Config2PhpHelperInterface;
use Drupal\config2php\Service\Config2PhpHelperTrait;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\service\CachedStorageTrait;
use Drupal\service\EntityTypeManagerTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for exporting a single configuration file.
 *
 * @phpstan-ignore-next-line
 */
class Config2PhpForm extends ConfigSingleExportForm {

  use CachedStorageTrait;
  use Config2PhpHelperTrait;
  use EntityTypeManagerTrait;

  /**
   * The beginning of the message about hidden keys.
   *
   * @var string
   */
  protected const PREFIX = 'Hidden key';

  /**
   * The end of the message about hidden keys.
   *
   * @var string
   */
  protected const SUFFIX = ': %keys.';

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return parent::create($container)->addConfig2PhpHelper($container);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'config2php_form';
  }

  /**
   * {@inheritdoc}
   */
  public function updateExport(...$arguments): array {
    if ($arguments[1] instanceof FormStateInterface) {
      foreach (Config2PhpHelperInterface::FIELDS as $field) {
        [, $variable] = explode('_', $field);
        $$variable = $arguments[1]->getValue($field);
      }
    }
    else {
      [$type, $name] = array_slice($arguments, 1, 2);
    }

    if (
      !empty($type) &&
      $type !== 'system.simple' &&
      ($definition = $this->entityTypeManager()->getDefinition($type)) !== NULL
    ) {
      $name = "{$definition->getConfigPrefix()}.$name";
    }

    $field = $arguments[0]['export'];

    if (!$this->configStorage()->exists($name)) {
      $field['#value'] = $field['#description'] = NULL;

      return $field;
    }

    $data = $this->configStorage()->read($name);

    $settings = $name === Config2PhpSettingsForm::NAME
      ? $data : $this->configStorage()->read(Config2PhpSettingsForm::NAME);

    if (!empty($settings['exclude'])) {
      $keys = array_intersect(array_keys($data), $settings['exclude']);

      foreach ($settings['exclude'] as $key) {
        unset($data[$key]);
      }
    }

    $field['#value'] = $this->config2PhpHelper()->convert($data);

    $field['#description'] = $this->t('Configuration name: %name', [
      '%name' => $name,
    ]);

    if (!empty($keys)) {
      $suffix = '';
      $arguments = ['%keys' => implode(', ', $keys)];

      if (($url = Url::fromRoute(Config2PhpSettingsForm::NAME))->access()) {
        $suffix = ' (it can be changed @link)';

        $arguments['@link'] = Link::fromTextAndUrl($this->t('here'), $url)
          ->toString();
      }

      $field['#description'] .= '<br />' . $this->formatPlural(
        count($keys),
        static::PREFIX . $suffix . static::SUFFIX,
        static::PREFIX . "s$suffix" . static::SUFFIX,
        $arguments,
      );
    }

    return $field;
  }

}
