<?php

namespace Drupal\config_enforce\Element;

use Drupal\config_enforce\EnforcedConfig;
use Drupal\config_enforce\Template\ConfigEnforceAttributesInterface;
use Drupal\Core\Render\Attribute\RenderElement;
use Drupal\Core\Render\Element\RenderElementBase;

/**
 * Provides a config enforce indicator element.
 */
#[RenderElement('config_enforce_indicator')]
class ConfigEnforceIndicator extends RenderElementBase {

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    return [
      '#pre_render' => [
        [$this, 'preRenderIndicator'],
      ],
      '#attached' => [
        'library' => ['config_enforce/config-enforce'],
      ],
    ];
  }

  /**
   * Pre-render callback: Renders a single config enforce indicator.
   *
   * @param array $element
   *   A structured array including the following keys:
   *   - #config_name: The name of the config object.
   * @param bool $displayUnenforced
   *   Whether to display unenforced config objects.
   *
   * @return array
   *   The passed-in element containing a rendered indicator in 'indicator'.
   */
  public function preRenderIndicator($element, $displayUnenforced = FALSE) {
    $config_name = $element['#config_name'];
    $isEnforced = EnforcedConfig::isEnforced($config_name);
    // Return early if the config is not enforced, and we are not displaying
    // unenforced configs (which will be the case unless Config Enforce Devel
    // is installed).
    if (!$isEnforced && !$displayUnenforced) return $element;

    $enforcementLevel = EnforcedConfig::getEnforcementLevel($config_name);

    $element['indicator'] = [
      '#type' => 'details',
      '#title' => [
        // This element is used to add the Config Enforce icon.
        '#type'       => 'html_tag',
        '#tag'        => 'span',
        '#attributes' => ['class' => ['config-enforce-container__summary-content']],
        '#value'      => $this->t(
          'Enforcement settings for: %config', ['%config' => $config_name]
        ),
      ],
      '#open' => FALSE,
      '#attributes' => [
        'class' => $this->getRowClasses($config_name),
        // These make it easier to identify which configuration name this item
        // represents in the front-end, whether it's currently enforced, and
        // what the enforcement level is.
        //
        // @see \Drupal\Tests\config_enforce\Functional\ConfigEnforceUiAssertTestTrait
        //   Initial use case is for this to assert whether or not a
        //   configuration item is listed on a given page and/or what its
        //   enforcement status is. Other use cases may be added in the future.
        ConfigEnforceAttributesInterface::CONFIG_DATA_ATTR_NAME =>
          $config_name,
        ConfigEnforceAttributesInterface::CONFIG_DATA_ATTR_STATUS =>
          $isEnforced ? 'true' : 'false',
        ConfigEnforceAttributesInterface::CONFIG_DATA_ATTR_ENFORCEMENT_LEVEL =>
          $enforcementLevel,
      ],
    ];

    $element['indicator']['config_enforce_enabled'] = [
      '#type' => 'item',
      '#title' => $this->t('Config enforce'),
      '#markup' => $isEnforced ? $this->t('Enabled') : $this->t('Disabled'),
    ];

    if ($isEnforced) {
      $element['indicator']['target_module'] = [
        '#type' => 'item',
        '#title' => $this->t('Module'),
        '#markup' => EnforcedConfig::getTargetModule($config_name),
      ];

      $element['indicator']['config_directory'] = [
        '#type' => 'item',
        '#title' => $this->t('Config directory'),
        '#markup' => EnforcedConfig::getConfigDirectory($config_name),
      ];

      $element['indicator']['config_enforce_path'] = [
        '#type' => 'item',
        '#title' => $this->t('Config file path'),
        '#markup' => EnforcedConfig::getConfigFilePath($config_name),
      ];

      $element['indicator']['enforcement_level'] = [
        '#type' => 'item',
        '#title' => $this->t('Enforcement level'),
        '#markup' => EnforcedConfig::getEnforcementLevelLabel($config_name),
      ];
    }

    return $element;
  }

  /**
   * Return row classes to allow for better styling.
   */
  protected function getRowClasses($config_name) {
    $classes = [];
    if (EnforcedConfig::isEnforced($config_name)) {
      $classes[] = 'enforced';
      $classes[] = 'level-' . EnforcedConfig::getEnforcementLevel($config_name);
    }
    return $classes;
  }

}
