<?php

declare(strict_types=1);

namespace Drupal\config_enforce_devel\Form;

use Drupal\config_enforce\ConfigEnforcer;
use Drupal\config_enforce\EnforcedConfig;
use Drupal\config_enforce_devel\EnforcedConfigCollection;
use Drupal\config_enforce_devel\TargetModuleCollection;
use Drupal\Core\Ajax\AjaxFormHelperTrait;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseDialogCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Config\InstallStorage;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Defines a settings form for the Config Enforce Devel module.
 */
class ConfigEnforceForm extends FormBase {

  use AjaxFormHelperTrait;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'config_enforce_form';
  }

  /**
   * Returns the title for managing enforcement settings for a piece of config.
   */
  public function getTitle(string $config_name) {
    return $this->t('Enforcement settings for: %config_name', ['%config_name' => $config_name]);
  }

  /**
   * {@inheritdoc}
   */
  protected function successfulAjaxSubmit(array $form, FormStateInterface $form_state) {
    // Reset static caches so that when we re-render the indicator, we get the
    // fresh enforcement settings.
    drupal_static_reset('Drupal\config_enforce\EnforcedConfigCollection::getAllEnforcedConfigs');

    $response = new AjaxResponse();
    $response->addCommand(new CloseDialogCommand('#drupal-off-canvas'));
    $response->addCommand(new MessageCommand($form_state->getTemporaryValue('status_message')));

    $config_name = $this->getConfigNameFromFormState($form_state);
    $indicator = [
      '#type' => 'config_enforce_indicator',
      '#config_name' => $config_name,
    ];
    $response->addCommand(new ReplaceCommand('[data-config-enforce-config-name="' . $config_name . '"]', $indicator));


    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $config_name = NULL) {
    // We want to preserve the hierarchy of form elements when getting form
    // values, in particular when dealing with dependencies.
    $form['#tree'] = TRUE;
    // TODO Check that $config_name is not a registry, and output an error and
    // redirect.
    // TODO Check that config_name is valid?
    $configIsEnforced = EnforcedConfig::isEnforced($config_name);

    $targetModuleCollection = new TargetModuleCollection();
    $form['config_enforce_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enforce config'),
      '#description' => $this->t("Enable enforcement of the config from this form."),
      '#default_value' => $configIsEnforced,
    ];

    $form['target_module'] = [
      '#type' => 'select',
      '#title' => $this->t('Target module'),
      '#description' => $this->t("The module to which config should be written."),
      '#options' => $targetModuleCollection->getTargetModuleLabels(),
      '#default_value' => $this->getDefaultValueTargetModule($config_name),
      '#states' => [
        'visible' => [
          ':input[name="config_enforce_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['config_directory'] = [
      '#type' => 'select',
      '#title' => $this->t('Directory'),
      '#description' => $this->t("The config directory to which config should be written."),
      '#options' => $this->getAvailableConfigDirectories(),
      '#default_value' => $this->getDefaultValueConfigDirectory($config_name),
      '#states' => [
        'visible' => [
          ':input[name="config_enforce_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    if ($configIsEnforced) {
      $form['config_enforce_path'] = [
        '#type' => 'container',
        '#states' => [
          'visible' => [
            ':input[name="config_enforce_enabled"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form['config_enforce_path']['markup'] = [
        '#type' => 'item',
        '#title' => $this->t('Config file path'),
        '#markup' => EnforcedConfig::getConfigFilePath($config_name),
      ];
    }

    $form['enforcement_level'] = [
      '#type' => 'select',
      '#title' => $this->t('Enforcement level'),
      '#description' => $this->t("The level of enforcement to apply to config objects."),
      '#options' => ConfigEnforcer::getEnforcementLevels(),
      '#default_value' => $this->getDefaultValueEnforcementLevel($config_name),
      '#states' => [
        'visible' => [
          ':input[name="config_enforce_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $this->addDependencyFields($form, $config_name);

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#button_type' => 'primary',
        '#value' => $this->t('Save')
      ],
    ];
    if ($this->isAjax()) {
      $form['actions']['submit']['#ajax']['callback'] = '::ajaxSubmit';
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config_name = $this->getConfigNameFromFormState($form_state);
    $isEnforced = EnforcedConfig::isEnforced($config_name);
    $enforcedConfigCollection = new EnforcedConfigCollection();

    $enableEnforcement = $form_state->getValue('config_enforce_enabled');
    // If the config is being unenforced, delete it.
    if ($isEnforced && !$enableEnforcement) {
      $enforcedConfig = EnforcedConfig::getEnforcedConfig($config_name);
      $enforcedConfigCollection
        ->deleteEnforcedConfigs([$config_name => $enforcedConfig]);
      $statusMessage = $this->t('Unenforced the configuration for %config_name.', ['%config_name' => $config_name]);
    }
    else {
      $enforcement_settings = [
        'target_module' => $form_state->getValue('target_module'),
        'config_directory' => $form_state->getValue('config_directory'),
        'enforcement_level' => $form_state->getValue('enforcement_level'),
        // TODO: Pass this in when enforcing new config as a query string or
        // similar (when available).
        // 'config_form_uri' => $this->getConfigFormUri(),
      ];

      $dependencies = [];
      if ($form_state->getValue('enforce_dependencies')) {
        // Build list of dependencies that are checked.
        $dependencies = array_keys(array_filter($form_state->getValue('dependencies')));
      }

      $this->saveEnforcedConfig($config_name, $enforcement_settings, $dependencies);
      if ($isEnforced) {
        $statusMessage = $this->t('Updated the enforcement settings for %config_name.', ['%config_name' => $config_name]);
      }
      else {
        $statusMessage = $this->t('Enforced the configuration for %config_name.', ['%config_name' => $config_name]);
      }
    }

    if ($this->isAjax()) {
      // Set the status message as a temporary value so that it can be retrieved
      // by successfulAjaxSubmit().
      $form_state->setTemporaryValue('status_message', $statusMessage);
    } else {
      $this->messenger()->addStatus($statusMessage);
    }
  }

  /**
   * Gets the config name from the form state.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return string
   *   The name of the config.
   */
  protected function getConfigNameFromFormState(FormStateInterface $form_state) {
    $formArgs = $form_state->getBuildInfo()['args'];
    if (!$formArgs) {
      // TODO Log an error, display an error.
      return;
    }

    return reset($formArgs);
  }

  /**
   * Creates or updates enforced configs.
   *
   * @param string $config_name
   *   The name of the config entity managed by this enforce form.
   * @param array $enforcement_settings
   *   An array of enforcement settings to be saved to the registry.
   * @param string[] $dependencies
   *   An array of config names that are dependencies of $config_name.
   */
  protected function saveEnforcedConfig(string $config_name, array $enforcement_settings, array $dependencies = []): void {
    $configs = [
      'create' => [],
      'update' => [],
    ];
    $key = EnforcedConfig::isEnforced($config_name) ? 'update' : 'create';
    $configs[$key][$config_name] = $enforcement_settings;

    if ($dependencies) {
      // We don't want to set config_form_uri for dependencies, it's only
      // relevant for the config we are directly enforcing.
      unset($enforcement_settings['config_form_uri']);
      foreach ($dependencies as $dependency) {
        $dependency_key = EnforcedConfig::isEnforced($dependency) ? 'update' : 'create';
        $configs[$dependency_key][$dependency] = $enforcement_settings;
      }
    }

    $enforcedConfigCollection = new EnforcedConfigCollection();
    $enforcedConfigCollection
      ->createEnforcedConfigs($configs['create']);
    $enforcedConfigCollection
      ->updateEnforcedConfigs($configs['update']);
  }

  /**
   * Returns a list of directories for writing config.
   */
  protected function getAvailableConfigDirectories() {
    // TODO Look this up in the storage service or make this a hook.
    return [
      InstallStorage::CONFIG_INSTALL_DIRECTORY => 'Install',
      InstallStorage::CONFIG_OPTIONAL_DIRECTORY => 'Optional',
    ];
  }

  /**
   * Gets the default value for the target module field.
   *
   * @param string $config_name
   *   The name of the config entity managed by this enforce form.
   *
   * @return string
   *   The default value for the field.
   */
  protected function getDefaultValueTargetModule(string $config_name) {
    // Get the value from the enforced config when available.
    if (EnforcedConfig::isEnforced($config_name)) {
      return EnforcedConfig::getTargetModule($config_name);
    }

    // If the config has not been enforced yet, return the default target
    // module.
    $targetModuleCollection = new TargetModuleCollection();
    return $targetModuleCollection->getDefaultTargetModule();
  }

  /**
   * Gets the default value for the config directory form field.
   *
   * @param string $config_name
   *   The name of the config entity managed by this enforce form.
   *
   * @return string
   *   The default value for the field.
   */
  protected function getDefaultValueConfigDirectory(string $config_name) {
    // Get the value from the enforced config when available.
    if (EnforcedConfig::isEnforced($config_name)) {
      return EnforcedConfig::getConfigDirectory($config_name);
    }

    // If the config has not been enforced yet, return the default setting from
    // Config Enforce Devel config.
    $defaults = $this->config('config_enforce_devel.settings')->get('defaults');
    $key = 'config_directory';
    if (is_array($defaults) && array_key_exists($key, $defaults)) {
      return $defaults[$key];
    }

    // If not configured in Config Enforce Devel, we default to the optional
    // config directory.
    return InstallStorage::CONFIG_OPTIONAL_DIRECTORY;
  }

  /**
   * Gets the default value for the enforcement level field.
   *
   * @param string $config_name
   *   The name of the config entity managed by this enforce form.
   *
   * @return string
   *   The default value for the field.
   */
  protected function getDefaultValueEnforcementLevel(string $config_name) {
    // Get the value from the enforced config when available.
    if (EnforcedConfig::isEnforced($config_name)) {
      return EnforcedConfig::getEnforcementLevel($config_name);
    }

    // If the config has not been enforced yet, return the default setting from
    // Config Enforce Devel config.
    $defaults = $this->config('config_enforce_devel.settings')->get('defaults');
    $key = 'enforcement_level';
    if (is_array($defaults) && array_key_exists($key, $defaults)) {
      return $defaults[$key];
    }

    // If not configured in Config Enforce Devel, we default to read only.
    return ConfigEnforcer::CONFIG_ENFORCE_READONLY;
  }

  /**
   * Gets the default value for the enforce dependencies field.
   *
   * @return bool
   *   TRUE to enforce dependencies, otherwise FALSE.
   */
  protected function getDefaultValueEnforceDependencies(): bool {
    // Return the default setting from Config Enforce Devel config.
    $defaults = $this->config('config_enforce_devel.settings')->get('defaults');
    $key = 'enforce_dependencies';
    if (is_array($defaults) && array_key_exists($key, $defaults)) {
      return (bool) $defaults[$key];
    }

    // If not configured in Config Enforce Devel, we default to TRUE (enforce
    // dependencies).
    return TRUE;
  }

  /**
   * Returns a list of all config objects that the current config object depends upon.
   *
   * Recurses to get all dependencies.
   *
   * @param string $config_name
   *   The name of the config entity managed by this enforce form.
   * @param array $known_deps
   *   An array of already-found dependencies, to avoid infinite recursion.
   *
   * @return string[]
   *   An array of all dependencies for $config_name.
   */
  protected function getConfigDependencies(string $config_name, array $known_deps = []): array {
    $all_deps = $this->config($config_name)->get('dependencies');
    if (empty($all_deps)) return [];
    $config_deps = array_key_exists('config', $all_deps) ? $all_deps['config'] : [];
    // Compile a list of known dependencies, so that we can avoid infinite recursion below.
    $known_deps = array_merge($known_deps, $config_deps);
    foreach ($config_deps as $config_dep) {
      // Recurse to ensure we're finding all dependencies.
      $new_deps = $this->getConfigDependencies($config_dep, $known_deps);
      // Strip out any dependencies that we've already found.
      $new_deps = array_diff($new_deps, $known_deps);
      $config_deps = array_merge($config_deps, $new_deps);
    }
    return $config_deps;
  }

  /**
   * Adds a checkbox to control whether to enforce dependent config objects.
   *
   * Also adds checkboxes for individual dependent config objects.
   */
  protected function addDependencyFields(array &$form, string $config_name) {
    $dependencies = $this->getConfigDependencies($config_name);
    if (!count($dependencies) > 0) return;

    $form['enforce_dependencies'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enforce dependencies'),
      '#description' => $this->t("Generate enforcement settings for the config objects upon which this one depends."),
      '#default_value' => $this->getDefaultValueEnforceDependencies(),
      '#states' => [
        'visible' => [
          ':input[name="config_enforce_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['dependencies'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="enforce_dependencies"]' => ['checked' => TRUE],
          ':input[name="config_enforce_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $enforced_dependencies = 0;
    foreach ($dependencies as $dependency) {
      $enforced = EnforcedConfig::isEnforced($dependency);
      if ($enforced) $enforced_dependencies++;

      $form['dependencies'][$dependency] = [
        '#type' => 'checkbox',
        '#title' => $dependency . ($enforced ? $this->t(' (already enforced)') : ''),
        '#default_value' => FALSE,
        '#states' => [
          'checked' => [
            ':input[name="enforce_dependencies"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    if ($enforced_dependencies > 0) {
      $this->disableEnforcedDependencies($form, $dependencies);
    }
  }

  /**
   * Disables checkboxes for dependencies that are already enforced, but allow for them to be overridden.
   *
   * Also adds an "Allow overriding..." checkbox.
   */
  protected function disableEnforcedDependencies(array &$form, array $dependencies) {
    $form['override_existing_dependencies'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow overriding existing enforcement settings'),
      '#description' => $this->t('Allow existing enforcement settings for dependencies to be overridden by the settings from this form.'),
      '#default_value' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name="enforce_dependencies"]' => ['checked' => TRUE],
          ':input[name="config_enforce_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    foreach ($dependencies as $dependency) {
      if (!EnforcedConfig::isEnforced($dependency)) continue;
      $form['dependencies'][$dependency]['#states'] = [
        'enabled' => [
          ':input[name="override_existing_dependencies"]' => ['checked' => TRUE],
        ],
      ];
    }
  }

}
