<?php

namespace Drupal\config_enforce_devel\Element;

use Drupal\config_enforce\Element\ConfigEnforceIndicator;
use Drupal\config_enforce_devel\IgnoredConfigTrait;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\Attribute\RenderElement;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a config enforce devel indicator element.
 */
#[RenderElement('config_enforce_devel_indicator')]
class ConfigEnforceDevelIndicator extends ConfigEnforceIndicator implements ContainerFactoryPluginInterface {

  use IgnoredConfigTrait;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->configFactory = $container->get('config.factory');

    return $instance;
  }

  /**
   * Returns a read-only config object.
   */
  protected function config($name) {
    return $this->configFactory->get($name);
  }

  /**
   * {@inheritdoc}
   */
  public function preRenderIndicator($element, $displayUnenforced = TRUE) {
    $element = parent::preRenderIndicator($element, $displayUnenforced);
    $config_name = $element['#config_name'];
    // Indicate in the UI when config objects are ignored. Note that this is
    // only relevant in the Config Enforce Devel context, since Config Enforce
    // itself doesn't have a concept of ignored configuration.
    if ($this->isIgnored($config_name)) {
      $element['indicator']['config_enforce_enabled']['#markup'] = $this->t('Ignored');
    }

    // Add a button to manage enforcement settings.
    $element['indicator']["config_enforce:{$config_name}"] = [
      '#title' => $this->t("Manage settings"),
      '#type' => 'link',
      '#url' => Url::fromRoute('config_enforce_devel.config_edit', [
        'config_name' => $config_name,
      ]),
      '#attributes' => [
        'class' => [
          'use-ajax',
          'button',
        ],
        'data-dialog-type' => "dialog",
        'data-dialog-renderer' => "off_canvas",
        'data-dialog-options' => [
          '{"width":400}',
        ],
      ],
      '#options' => [
        'query' => \Drupal::destination()->getAsArray(),
      ],
      '#attached' => [
        'library' => 'core/drupal.dialog.ajax',
      ],
    ];

    return $element;
  }

}
