<?php

declare(strict_types=1);

namespace Drupal\config_enforce_devel;

use Drupal\config_enforce\EnforcedConfig;

/**
 * Defines a trait for functionality related to enforce-by-default.
 */
trait EnforceByDefaultTrait {

  use IgnoredConfigTrait;

  /**
   * Gets a read-only version of the given config object.
   *
   * @param string $name
   *   The name of the config object to get.
   */
  abstract protected function config($name);

  /**
   * Gets the config factory.
   *
   * @return \Drupal\Core\Config\ConfigFactoryInterface
   *   The configuration factory.
   */
  abstract protected function configFactory();

  /**
   * Determine whether configs should be enforced by default.
   */
  public function shouldConfigsBeEnforcedByDefault(): bool {
    $enforce_by_default = $this->config('config_enforce_devel.settings')->get('defaults.enforce_by_default');
    if (is_null($enforce_by_default)) {
      return TRUE;
    }

    return $enforce_by_default;
  }

  /**
   * Determine whether we should enforce a given config object.
   */
  protected function shouldConfigBeEnforced(string $config_name): bool {
    if (EnforcedConfig::isEnforced($config_name)) {
      return TRUE;
    }
    if ($this->isIgnored($config_name)) {
      return FALSE;
    }

    return $this->shouldConfigsBeEnforcedByDefault();
  }

  /**
   * Add given config object to ignore list.
   */
  protected function addConfigToIgnoreList(string $config_name): void {
    $ignored_configs = $this->getIgnoredConfigs();
    // Exit early if the config is already ignored.
    if (array_key_exists($config_name, $ignored_configs)) return;

    $ignored_configs[$config_name] = $config_name;

    $settings = $this->configFactory()->getEditable('config_enforce_devel.settings');
    $settings->set('ignored_configs', array_values($ignored_configs))->save();
  }

  /**
   * Remove given config object from ignore list.
   */
  protected function removeConfigFromIgnoreList(string $config_name): void {
    $ignored_configs = $this->getIgnoredConfigs();
    // Exit early if the config is not currently ignored.
    if (!array_key_exists($config_name, $ignored_configs)) return;

    unset($ignored_configs[$config_name]);

    $settings = $this->configFactory()->getEditable('config_enforce_devel.settings');
    $settings->set('ignored_configs', array_values($ignored_configs))->save();
  }

}
