<?php

declare(strict_types=1);

namespace Drupal\config_enforce_devel\Form;

use Drupal\Component\Utility\UrlHelper;
use Drupal\config_enforce\ConfigEnforcer;
use Drupal\config_enforce\EnforcedConfig;
use Drupal\config_enforce_devel\EnforceByDefaultTrait;
use Drupal\config_enforce_devel\EnforcedConfigCollection;
use Drupal\config_enforce_devel\SaveEnforcedConfigSettingsTrait;
use Drupal\config_enforce_devel\TargetModuleCollection;
use Drupal\Core\Ajax\AjaxFormHelperTrait;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\CloseDialogCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Config\InstallStorage;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a settings form for the Config Enforce Devel module.
 */
class ConfigEnforceForm extends FormBase {

  use AjaxFormHelperTrait;
  use SaveEnforcedConfigSettingsTrait;
  use EnforceByDefaultTrait;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->requestStack = $container->get('request_stack');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'config_enforce_form';
  }

  /**
   * Returns the title for managing enforcement settings for a piece of config.
   */
  public function getTitle(string $config_name) {
    return $this->t('Enforcement settings for: %config_name', ['%config_name' => $config_name]);
  }

  /**
   * {@inheritdoc}
   */
  protected function successfulAjaxSubmit(array $form, FormStateInterface $form_state) {
    // Reset static caches so that when we re-render the indicator, we get the
    // fresh enforcement settings.
    drupal_static_reset('Drupal\config_enforce\EnforcedConfigCollection::getAllEnforcedConfigs');

    $response = new AjaxResponse();
    $response->addCommand(new CloseDialogCommand('#drupal-off-canvas'));
    foreach ($form_state->getTemporaryValue('status_messages') as $statusMessage) {
      $response->addCommand(new MessageCommand($statusMessage, clear_previous: FALSE));
    }

    $config_name = $this->getConfigNameFromFormState($form_state);
    $indicator = [
      '#type' => 'config_enforce_devel_indicator',
      '#config_name' => $config_name,
    ];
    $response->addCommand(new ReplaceCommand('[data-config-enforce-config-name="' . $config_name . '"]', $indicator));


    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $config_name = NULL) {
    // We want to preserve the hierarchy of form elements when getting form
    // values, in particular when dealing with dependencies.
    $form['#tree'] = TRUE;
    // TODO Check that $config_name is not a registry, and output an error and
    // redirect.
    // TODO Check that config_name is valid?
    $configIsEnforced = EnforcedConfig::isEnforced($config_name);

    $enabledTitleSuffix = '';
    $enabledDescriptionSuffix = '';
    // If enforcing config by default, override title and description to be more
    // instructive.
    if ($this->shouldConfigsBeEnforcedByDefault()) {
      // Not yet enforced.
      if (!EnforcedConfig::isEnforced($config_name)) {
        $enabledTitleSuffix = $this->t('(not yet enforced)');
        $enabledDescriptionSuffix = $this->t("Config will be enforced by default. Uncheck to ignore this config.");
      }
      // Currently ignored.
      if ($this->isIgnored($config_name)) {
        $enabledTitleSuffix = $this->t('(currently ignored)');
        $enabledDescriptionSuffix = $this->t("Ignored config will not be enforced by default.");
      }
    }

    $targetModuleCollection = new TargetModuleCollection();
    $form['config_enforce_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enforce config @suffix', ['@suffix' => $enabledTitleSuffix]),
      '#description' => $this->t('Enable enforcement of the config from this form. @suffix', ['@suffix' => $enabledDescriptionSuffix]),
      '#default_value' => $this->shouldConfigBeEnforced($config_name),
    ];

    $form['target_module'] = [
      '#type' => 'select',
      '#title' => $this->t('Target module'),
      '#description' => $this->t("The module to which config should be written."),
      '#options' => $targetModuleCollection->getTargetModuleLabels(),
      '#default_value' => $this->getDefaultValueTargetModule($config_name),
      '#states' => [
        'visible' => [
          ':input[name="config_enforce_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['config_directory'] = [
      '#type' => 'select',
      '#title' => $this->t('Directory'),
      '#description' => $this->t("The config directory to which config should be written."),
      '#options' => $this->getAvailableConfigDirectories(),
      '#default_value' => $this->getDefaultValueConfigDirectory($config_name),
      '#states' => [
        'visible' => [
          ':input[name="config_enforce_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    if ($configIsEnforced) {
      $form['config_enforce_path'] = [
        '#type' => 'container',
        '#states' => [
          'visible' => [
            ':input[name="config_enforce_enabled"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form['config_enforce_path']['markup'] = [
        '#type' => 'item',
        '#title' => $this->t('Config file path'),
        '#markup' => EnforcedConfig::getConfigFilePath($config_name),
      ];
    }

    $form['enforcement_level'] = [
      '#type' => 'select',
      '#title' => $this->t('Enforcement level'),
      '#description' => $this->t("The level of enforcement to apply to config objects."),
      '#options' => ConfigEnforcer::getEnforcementLevels(),
      '#default_value' => $this->getDefaultValueEnforcementLevel($config_name),
      '#states' => [
        'visible' => [
          ':input[name="config_enforce_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $this->addDependencyFields($form, $config_name);

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#button_type' => 'primary',
        '#value' => $this->t('Save')
      ],
    ];
    if ($this->isAjax()) {
      $form['actions']['submit']['#ajax']['callback'] = '::ajaxSubmit';
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config_name = $this->getConfigNameFromFormState($form_state);
    $isEnforced = EnforcedConfig::isEnforced($config_name);
    $enableEnforcement = $form_state->getValue('config_enforce_enabled');
    $enforcedConfigCollection = new EnforcedConfigCollection();
    $statusMessages = [];

    if ($this->shouldConfigsBeEnforcedByDefault()) {
      if (!$enableEnforcement) {
        $this->addConfigToIgnoreList($config_name);
        $statusMessages[] = $this->t('Ignored the configuration for %config_name.', ['%config_name' => $config_name]);
      }
      else {
        $this->removeConfigFromIgnoreList($config_name);
      }
    }

    // If the config is being unenforced, delete it.
    if ($isEnforced && !$enableEnforcement) {
      $enforcedConfig = EnforcedConfig::getEnforcedConfig($config_name);
      $enforcedConfigCollection
        ->deleteEnforcedConfigs([$config_name => $enforcedConfig]);
      $statusMessages[] = $this->t('Unenforced the configuration for %config_name.', ['%config_name' => $config_name]);
    }
    else if ($enableEnforcement) {
      $enforcement_settings = [
        'target_module' => $form_state->getValue('target_module'),
        'config_directory' => $form_state->getValue('config_directory'),
        'enforcement_level' => $form_state->getValue('enforcement_level'),
        'config_form_uri' => $this->getConfigFormUriFromDestination($config_name),
      ];

      $dependencies = [];
      if ($form_state->getValue('enforce_dependencies')) {
        // Build list of dependencies that are checked.
        $dependencies = array_keys(array_filter($form_state->getValue('dependencies')));
      }

      $configs = $this->saveEnforcedConfigSettings($config_name, $enforcement_settings, $dependencies);
      foreach ($this->getOperationMessages($configs) as $message) {
        $statusMessages[] = $message;
      }
    }

    if ($this->isAjax()) {
      // Set the status messages as a temporary value so that they can be
      // retrieved by successfulAjaxSubmit().
      $form_state->setTemporaryValue('status_messages', $statusMessages);
    } else {
      foreach ($statusMessages as $statusMessage) {
        $this->messenger()->addStatus($statusMessage);
      }
    }
  }

  /**
   * Gets the config name from the form state.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return string
   *   The name of the config.
   */
  protected function getConfigNameFromFormState(FormStateInterface $form_state) {
    $formArgs = $form_state->getBuildInfo()['args'];
    if (!$formArgs) {
      // TODO Log an error, display an error.
      return;
    }

    return reset($formArgs);
  }

  /**
   * Returns a list of directories for writing config.
   */
  protected function getAvailableConfigDirectories() {
    // TODO Look this up in the storage service or make this a hook.
    return [
      InstallStorage::CONFIG_INSTALL_DIRECTORY => 'Install',
      InstallStorage::CONFIG_OPTIONAL_DIRECTORY => 'Optional',
    ];
  }

  /**
   * Adds a checkbox to control whether to enforce dependent config objects.
   *
   * Also adds checkboxes for individual dependent config objects.
   */
  protected function addDependencyFields(array &$form, string $config_name) {
    $dependencies = $this->getConfigDependencies($config_name);
    if (!count($dependencies) > 0) return;

    $form['enforce_dependencies'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enforce dependencies'),
      '#description' => $this->t("Generate enforcement settings for the config objects upon which this one depends."),
      '#default_value' => $this->getDefaultValueEnforceDependencies(),
      '#states' => [
        'visible' => [
          ':input[name="config_enforce_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['dependencies'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="enforce_dependencies"]' => ['checked' => TRUE],
          ':input[name="config_enforce_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $enforced_dependencies = 0;
    foreach ($dependencies as $dependency) {
      $enforced = EnforcedConfig::isEnforced($dependency);
      if ($enforced) $enforced_dependencies++;

      $form['dependencies'][$dependency] = [
        '#type' => 'checkbox',
        '#title' => $dependency . ($enforced ? $this->t(' (already enforced)') : ''),
        '#default_value' => FALSE,
        '#states' => [
          'checked' => [
            ':input[name="enforce_dependencies"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    if ($enforced_dependencies > 0) {
      $this->disableEnforcedDependencies($form, $dependencies);
    }
  }

  /**
   * Disables checkboxes for dependencies that are already enforced, but allow for them to be overridden.
   *
   * Also adds an "Allow overriding..." checkbox.
   */
  protected function disableEnforcedDependencies(array &$form, array $dependencies) {
    $form['override_existing_dependencies'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow overriding existing enforcement settings'),
      '#description' => $this->t('Allow existing enforcement settings for dependencies to be overridden by the settings from this form.'),
      '#default_value' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name="enforce_dependencies"]' => ['checked' => TRUE],
          ':input[name="config_enforce_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    foreach ($dependencies as $dependency) {
      if (!EnforcedConfig::isEnforced($dependency)) continue;
      $form['dependencies'][$dependency]['#states'] = [
        'enabled' => [
          ':input[name="override_existing_dependencies"]' => ['checked' => TRUE],
        ],
      ];
    }
  }

  /**
   * Gets the value for config_form_uri when adding/updating the registry.
   * 
   * @param string $config_name
   *   The config name.
   */
  protected function getConfigFormUriFromDestination(string $config_name) {
    $fallbackUri = NULL;
    // If the config is already enforced, start with the existing value from the
    // registry.
    if (EnforcedConfig::isEnforced($config_name)) {
      $fallbackUri = EnforcedConfig::getConfigFormUri($config_name);
    }

    $query = $this->requestStack->getCurrentRequest()->query;
    // Return a default value if no destination querystring is available.
    if (!$query->has('destination')) {
      return $fallbackUri;
    }

    $destination = $query->get('destination');
    // Return a default value if the querystring is an external URL.
    if (UrlHelper::isExternal($destination)) {
      return $fallbackUri;
    }

    // Return a default value if the querystring is the Enforced Configs form.
    if ($destination == Url::fromRoute('config_enforce_devel.enforced_configs')->toString()) {
      return $fallbackUri;
    }

    return $destination;
  }

}
