<?php

declare(strict_types=1);

namespace Drupal\config_enforce_devel\Form;

use Drupal\config_enforce\ConfigEnforcer;
use Drupal\config_enforce\Form\FormHelperTrait;
use Drupal\config_enforce_devel\EnforcedConfigCollection;
use Drupal\config_enforce_devel\IgnoredConfigTrait;
use Drupal\Core\Config\InstallStorage;
use Drupal\Core\Form\FormStateInterface;

/**
 * Defines a settings form for the Config Enforce Devel module.
 *
 * @internal
 */
trait DevelFormHelperTrait {

  use FormHelperTrait {
    setSharedFormProperties as defaultSetSharedFormProperties;
  }
  use IgnoredConfigTrait;

   /**
   * The enforced config collection.
   *
   * @var \Drupal\config_enforce_devel\EnforcedConfigCollection
   */
  protected $enforcedConfigCollection;

  /**
   * {@inheritdoc}
   */
  protected function setSharedFormProperties(array &$form, FormStateInterface &$form_state) {
    $this->defaultSetSharedFormProperties($form, $form_state);
    $this->enforcedConfigCollection = new EnforcedConfigCollection();
    return $this;
  }

  /**
   * Return an enforced config collection.
   */
  protected function getEnforcedConfigCollection() {
    return $this->enforcedConfigCollection;
  }

  /**
   * Return a list of directories for writing config.
   */
  protected function getAvailableConfigDirectories() {
    // @TODO Look this up in the storage service or make this a hook.
    return [
      InstallStorage::CONFIG_INSTALL_DIRECTORY => 'Install',
      InstallStorage::CONFIG_OPTIONAL_DIRECTORY => 'Optional',
    ];
  }

  /**
   * Return the basic definition for a target module field.
   */
  protected function getTargetModuleField() {
    return [
      '#type' => 'select',
      '#title' => $this->t('Target module'),
      '#description' => "The module to which config should be written.",
      // Used cached value generated by scanning for `enforced_configs`
      '#options' => $this
        ->getEnforcedConfigCollection()
        ->getTargetModuleCollection()
        ->getTargetModuleLabels(),
      '#default_value' => $this
        ->getEnforcedConfigCollection()
        ->getTargetModuleCollection()
        ->getDefaultTargetModule(),
    ];
  }

  /**
   * Return the basic definition for a config directory field.
   */
  protected function getConfigDirectoryField() {
    $defaults = $this->config('config_enforce_devel.settings')->get('defaults');
    $key = 'config_directory';
    $default = is_array($defaults) && array_key_exists($key, $defaults) ? $defaults[$key] : InstallStorage::CONFIG_OPTIONAL_DIRECTORY;
    return [
      '#type' => 'select',
      '#title' => $this->t('Directory'),
      '#description' => "The config directory to which config should be written.",
      '#options' => $this->getAvailableConfigDirectories(),
      '#default_value' => $default,
    ];
  }

  /**
   * Return the basic definition for an enforcement level field.
   */
  protected function getEnforcementLevelField() {
    $defaults = $this->config('config_enforce_devel.settings')->get('defaults');
    $key = 'enforcement_level';
    $default = is_array($defaults) && array_key_exists($key, $defaults) ? $defaults[$key] : ConfigEnforcer::CONFIG_ENFORCE_READONLY;
    return [
      '#type' => 'select',
      '#title' => $this->t('Enforcement level'),
      '#description' => "The level of enforcement to apply to config objects.",
      '#options' => ConfigEnforcer::getEnforcementLevels(),
      '#default_value' => $default,
    ];
  }

  /**
   * List unenforced config object names that have not been flagged for ignoring.
   */
  protected function getUnenforcedConfigs() {
    $unenforced_configs = $this->getAllUnenforcedConfigs();
    $ignored_configs = $this->getIgnoredConfigs();
    $filtered_configs = array_diff($unenforced_configs, $ignored_configs);
    return array_combine($filtered_configs, $filtered_configs);
  }

  /**
   * List all unenforced config object names.
   */
  protected function getAllUnenforcedConfigs() {
    $active_storage_configs = $this->configFactory()->listAll();
    $enforced_configs = array_keys($this->getEnforcedConfigCollection()->getEnforcedConfigs());
    $unenforced_configs = array_diff($active_storage_configs, $enforced_configs);
    return array_combine($unenforced_configs, $unenforced_configs);
  }

}
