<?php

declare(strict_types=1);

namespace Drupal\config_enforce_devel\Form;

use Drupal\config_enforce_devel\EnforceByDefaultTrait;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\config_enforce_devel\EnforcedConfig;
use Drupal\config_enforce_devel\Form\DevelFormHelperTrait;
use Drupal\config_enforce_devel\TargetModuleCollection;

/**
 * Defines a settings form for the Config Enforce Devel module.
 */
class SettingsForm extends ConfigFormBase {

  use DevelFormHelperTrait;
  use EnforceByDefaultTrait;

  const FORM_ID = 'config_enforce_devel_settings';

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'config_enforce_devel.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $this->setSharedFormProperties($form, $form_state);

    $this->addDefaultsFields();
    $this->addAvailableModulesField();
    $this->addIgnoredConfigsField();

    return parent::buildForm($this->form(), $this->formState());
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->setSharedFormProperties($form, $form_state);

    $this->config('config_enforce_devel.settings')
      ->set('defaults', $this->formState()->getValue('defaults'))
      ->set('ignored_configs', array_values($this->formState()->getValue('ignored_configs')))
      ->save();

    $available_modules = $this->formState()->getValue('available_modules');
    $target_modules = $this->getEnforcedConfigCollection()->getTargetModuleCollection();
    $target_modules
      ->ensureTargetModulesAreRegistered($available_modules);

    $default_target_module = $target_modules->getDefaultTargetModule();
    if (!in_array($default_target_module, $available_modules)) {
      $warning = $this->t('Default target module (@default) cannot be de-selected.', [
        '@default' => $default_target_module,
      ]);
      $this->messenger()->addWarning($warning);
    }

    return parent::submitForm($this->form(), $this->formState());
  }

  /**
   * Add field to select available modules.
   */
  protected function addAvailableModulesField() {
    $this->form()['target_modules'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#title' => $this->t('Target modules'),
    ];

    $this->form()['target_modules']['available_modules'] = [
      '#type' => 'multiselect',
      '#title' => $this->t('Select available target modules'),
      '#description' => $this->t('These modules will be available as target modules on config forms, to specify where configuration files should be written.<br />Note that you cannot de-select the default target module (see the defaults above).'),
      '#options' => $this
        ->getEnforcedConfigCollection()
        ->getTargetModuleCollection()
        ->getAllModules(),
      '#multiple' => TRUE,
      '#default_value' => $this
        ->getEnforcedConfigCollection()
        ->getTargetModuleCollection()
        ->getTargetModuleNames(),
    ];
  }

  /**
   * Add fields to define defaults for enforcement settings fields.
   */
  protected function addDefaultsFields() {
    $this->form()['defaults'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#tree' => TRUE,
      '#title' => $this->t('Enforcement defaults'),
    ];

    $this->form()['defaults']['enforce_by_default'] = [
      '#type' => 'checkbox',
      '#default_value' => $this->shouldConfigsBeEnforcedByDefault(),
      '#title' => $this->t('Enforce all configs by default'),
      '#description' => $this->t('<strong>Strongly recommended.</strong> When saving a config form, the associated configuration will be enforced unless specifically ignored.'),
    ];
    $this->form()['defaults']['target_module'] = $this->getTargetModuleField();
    $this->form()['defaults']['config_directory'] = $this->getConfigDirectoryField();
    $this->form()['defaults']['enforcement_level'] = $this->getEnforcementLevelField();
    $this->form()['defaults']['enforce_dependencies'] = $this->getEnforceDependenciesField();
  }

  /**
   * Add field to select available modules.
   */
  protected function addIgnoredConfigsField() {
    $this->form()['unenforced_configs'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#title' => $this->t('Ignored configs'),
    ];

    $this->form()['unenforced_configs']['ignored_configs'] = [
      '#type' => 'multiselect',
      '#title' => $this->t('Select config to ignore'),
      // TODO: Linkify 'Generate from active storage' once we make it
      // a standalone form.
      '#description' => $this->t('These configs will not appear in lists of unenforced configs (e.g. when using <em>Generate from active storage</em>).<br />When <em>Enforce all configs by default
</em> is enabled, unenforcing any given config will add it to this list.'),
      '#options' => $this->getAllUnenforcedConfigs(),
      '#multiple' => TRUE,
      '#default_value' => $this->getIgnoredConfigs(),
    ];
  }

  /**
   * Add a checkbox to control whether to enforce dependent config objects.
   */
  protected function getEnforceDependenciesField() {
    $defaults = $this->config('config_enforce_devel.settings')->get('defaults');
    $key = 'enforce_dependencies';
    $default = is_array($defaults) && array_key_exists($key, $defaults) ? $defaults[$key] : TRUE;
    return [
      '#type' => 'checkbox',
      '#title' => 'Enforce dependencies',
      '#description' => $this->t("<strong>Strongly recommended.</strong> Generate enforcement settings for the config objects upon which this one depends."),
      '#default_value' => $default,
    ];
  }

}
