<?php

declare(strict_types=1);

namespace Drupal\config_enforce_devel\FormHandler;

use Drupal\config_enforce\ConfigResolver;
use Drupal\config_enforce\EnforcedConfig;
use Drupal\config_enforce_devel\EnforceByDefaultTrait;
use Drupal\config_enforce_devel\SaveEnforcedConfigSettingsTrait;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Render\Element;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Add a submit handler to all config forms.
 */
class DevelEnforceFormHandler {

  use DependencySerializationTrait;
  use EnforceByDefaultTrait;
  use MessengerTrait;
  use SaveEnforcedConfigSettingsTrait;
  use StringTranslationTrait;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The URL generator.
   *
   * @var \Drupal\Core\Routing\UrlGeneratorInterface
   */
  protected $urlGenerator;

  /**
   * Constructs a DevelEnforceFormHandler object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Routing\UrlGeneratorInterface $url_generator
   *   The URL generator.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    UrlGeneratorInterface $url_generator,
  ) {
    $this->configFactory = $config_factory;
    $this->urlGenerator = $url_generator;
  }

  /**
   * Returns a read-only config object.
   */
  protected function config($name) {
    return $this->configFactory->get($name);
  }

  /**
   * Gets the config factory.
   *
   * When accessing configuration values, use $this->config(). Only use this
   * when the config factory needs to be manipulated directly.
   *
   * @return \Drupal\Core\Config\ConfigFactoryInterface
   *   The configuration factory.
   */
  protected function configFactory() {
    return $this->configFactory;
  }

  /**
   * Method to call from implementations of hook_alter().
   */
  public function alter(&$form, FormStateInterface $form_state) {
    // Only operate on config forms.
    $configResolver = new ConfigResolver($form_state);
    // We want to be able to immediately enforce new config entities as they are
    // created, so we also want to include config entities that have not yet
    // been saved.
    $hasConfigEntity = $configResolver->hasConfigEntity(includeNewEntities: TRUE);
    if (!$configResolver->isAnEnforceableForm() && !$hasConfigEntity) return;

    // Attach our submit handler directly to the submit button when present,
    // otherwise it won't run in those cases.
    // @see \Drupal\Core\Form\FormSubmitter::executeSubmitHandlers()
    if (isset($form['actions']['submit']['#submit'])) {
      $form['actions']['submit']['#submit'][] = [$this, 'enforceByDefaultSubmit'];
    }
    else {
      $form['#submit'][] = [$this, 'enforceByDefaultSubmit'];
    }
  }

  /**
   * Submission handler to enforce configuration when saving associated config form.
   *
   * @param array $form
   *   Form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  public function enforceByDefaultSubmit(array $form, FormStateInterface $form_state) {
    // Return early if the global enforce configs by default setting has been
    // disabled. We check here rather than conditionally attaching the submit
    // handler so that "stale" forms behave as expected after changing the
    // global setting.
    if (!$this->shouldConfigsBeEnforcedByDefault()) {
      return;
    }
    // TODO: Add separate behaviour somewhere to update config_form_uri if
    // that has changed from what we saved in the registry. This should not be
    // tied to enforce by default, because if a user is only using ConfigEnforceForm,
    // the config_form_uri will never be updated.

    $configResolver = new ConfigResolver($form_state);
    $enforceable_configs = $configResolver->getConfigNames();
    if (is_null($enforceable_configs)) return;

    $all_configs = [];
    foreach ($enforceable_configs as $config_name) {
      if ($this->isIgnored($config_name)) continue;

      $dependencies = [];
      if ($this->getDefaultValueEnforceDependencies()) {
        $dependencies = $this->getConfigDependencies($config_name);
        // Remove any dependencies that are already enforced, we don't want to
        // change their enforcement settings. If we pass all dependencies to
        // saveEnforcedConfigSettings(), then all dependencies will be updated
        // to match the enforcement settings of $config_name.
        foreach ($dependencies as $index => $dependency) {
          if (EnforcedConfig::isEnforced($dependency)) {
            unset($dependencies[$index]);
          }
        }
      }

      // If the config is already enforced, and there are no new dependencies to
      // enforce, then skip to the next $config_name.
      if (EnforcedConfig::isEnforced($config_name) && !$dependencies) {
        continue;
      }

      $configs = $this->saveEnforcedConfigSettings($config_name, $this->getEnforcementSettings($config_name), $dependencies);
      // Merge all config operations into a single array.
      $all_configs = array_merge_recursive($all_configs, $configs);
    }
    foreach ($this->getOperationMessages($all_configs) as $message) {
      $this->messenger()->addStatus($message);
    }
  }

  /**
   * Gets enforcement settings in the context of a config form submit handler.
   *
   * @param string $config_name
   *   The config name.
   * 
   * @return array
   *   An array of enforcement settings.
   */
  protected function getEnforcementSettings(string $config_name): array {
    return [
      'target_module' => $this->getDefaultValueTargetModule($config_name),
      'config_directory' => $this->getDefaultValueConfigDirectory($config_name),
      'enforcement_level' => $this->getDefaultValueEnforcementLevel($config_name),
      'config_form_uri' => $this->urlGenerator->generateFromRoute('<current>'),
    ];
  }

}
