<?php

declare(strict_types=1);

namespace Drupal\config_enforce_devel;

use Drupal\config_enforce\ConfigEnforcer;
use Drupal\config_enforce\EnforcedConfig;
use Drupal\Core\Config\InstallStorage;

/**
 * Defines a trait to save enforced configuration.
 */
trait SaveEnforcedConfigSettingsTrait {

  /**
   * Gets a read-only version of the given config object.
   *
   * @param string $name
   *   The name of the config object to get.
   */
  abstract protected function config($name);

  /**
   * Creates or updates enforced config settings.
   *
   * @param string $config_name
   *   The name of the config entity managed by this enforce form.
   * @param array $enforcement_settings
   *   An array of enforcement settings to be saved to the registry.
   * @param string[] $dependencies
   *   An array of config names that are dependencies of $config_name.
   *
   * @return array[]
   *   An array with the following keys:
   *   - create: Array of enforcement settings for configs that were enforced,
   *   keyed by config name.
   *   - update: Array of updated enforcement settings for configs, keyed by
   *   config name.
   */
  protected function saveEnforcedConfigSettings(string $config_name, array $enforcement_settings, array $dependencies = []): array {
    $configs = [
      'create' => [],
      'update' => [],
    ];

    if ($this->shouldPerformOperation($config_name, $enforcement_settings)) {
      $key = EnforcedConfig::isEnforced($config_name) ? 'update' : 'create';
      $configs[$key][$config_name] = $enforcement_settings;
    }

    if ($dependencies) {
      // We don't want to set config_form_uri for dependencies, it's only
      // relevant for the config we are directly enforcing.
      unset($enforcement_settings['config_form_uri']);
      foreach ($dependencies as $dependency) {
        if (!$this->shouldPerformOperation($dependency, $enforcement_settings)) {
          continue;
        }

        $dependency_key = EnforcedConfig::isEnforced($dependency) ? 'update' : 'create';
        $configs[$dependency_key][$dependency] = $enforcement_settings;
      }
    }

    $enforcedConfigCollection = new EnforcedConfigCollection();
    $enforcedConfigCollection
      ->createEnforcedConfigs($configs['create']);
    $enforcedConfigCollection
      ->updateEnforcedConfigs($configs['update']);

    return $configs;
  }

  /**
   * Determines whether to perform an operation on the given config.
   *
   * @param string $config_name
   *   The name of the config entity.
   * @param array $enforcement_settings
   *   An array of enforcement settings.
   *
   * @return bool
   *   TRUE if the operation should be performed, FALSE if it should be skipped.
   */
  protected function shouldPerformOperation(string $config_name, array $enforcement_settings): bool {
    $isEnforced = EnforcedConfig::isEnforced($config_name);
    // When creating, we always perform the operation.
    if (!$isEnforced) {
      return TRUE;
    }

    // When updating, only update if the enforcement settings have changed
    // ($enforcement_settings differs from what we already have in the config
    // enforce registry).
    // First, get the existing settings from the config enforce registry.
    $existingSettings = EnforcedConfig::getEnforcedConfig($config_name);
    // Remove any keys that are not present in $enforcement_settings, so that
    // we can compare the two arrays directly.
    $existingSettingsToCompare = array_intersect_key($existingSettings, $enforcement_settings);

    return $existingSettingsToCompare != $enforcement_settings;
  }

  /**
   * Gets the default value for the target module field.
   *
   * @param string $config_name
   *   The name of the config entity managed by this enforce form.
   *
   * @return string
   *   The default value for the field.
   */
  protected function getDefaultValueTargetModule(string $config_name) {
    // Get the value from the enforced config when available.
    if (EnforcedConfig::isEnforced($config_name)) {
      return EnforcedConfig::getTargetModule($config_name);
    }

    // If the config has not been enforced yet, return the default target
    // module.
    $targetModuleCollection = new TargetModuleCollection();
    return $targetModuleCollection->getDefaultTargetModule();
  }

  /**
   * Gets the default value for the config directory form field.
   *
   * @param string $config_name
   *   The name of the config entity managed by this enforce form.
   *
   * @return string
   *   The default value for the field.
   */
  protected function getDefaultValueConfigDirectory(string $config_name) {
    // Get the value from the enforced config when available.
    if (EnforcedConfig::isEnforced($config_name)) {
      return EnforcedConfig::getConfigDirectory($config_name);
    }

    // If the config has not been enforced yet, return the default setting from
    // Config Enforce Devel config.
    $defaults = $this->config('config_enforce_devel.settings')->get('defaults');
    $key = 'config_directory';
    if (is_array($defaults) && array_key_exists($key, $defaults)) {
      return $defaults[$key];
    }

    // If not configured in Config Enforce Devel, we default to the optional
    // config directory.
    return InstallStorage::CONFIG_OPTIONAL_DIRECTORY;
  }

  /**
   * Gets the default value for the enforcement level field.
   *
   * @param string $config_name
   *   The name of the config entity managed by this enforce form.
   *
   * @return string
   *   The default value for the field.
   */
  protected function getDefaultValueEnforcementLevel(string $config_name) {
    // Get the value from the enforced config when available.
    if (EnforcedConfig::isEnforced($config_name)) {
      return EnforcedConfig::getEnforcementLevel($config_name);
    }

    // If the config has not been enforced yet, return the default setting from
    // Config Enforce Devel config.
    $defaults = $this->config('config_enforce_devel.settings')->get('defaults');
    $key = 'enforcement_level';
    if (is_array($defaults) && array_key_exists($key, $defaults)) {
      return $defaults[$key];
    }

    // If not configured in Config Enforce Devel, we default to read only.
    return ConfigEnforcer::CONFIG_ENFORCE_READONLY;
  }

  /**
   * Gets the default value for the enforce dependencies field.
   *
   * @return bool
   *   TRUE to enforce dependencies, otherwise FALSE.
   */
  protected function getDefaultValueEnforceDependencies(): bool {
    // Return the default setting from Config Enforce Devel config.
    $defaults = $this->config('config_enforce_devel.settings')->get('defaults');
    $key = 'enforce_dependencies';
    if (is_array($defaults) && array_key_exists($key, $defaults)) {
      return (bool) $defaults[$key];
    }

    // If not configured in Config Enforce Devel, we default to TRUE (enforce
    // dependencies).
    return TRUE;
  }

  /**
   * Returns a list of all config objects that a given config object depends upon.
   *
   * Recurses to get all dependencies.
   *
   * @param string $config_name
   *   The name of the config entity managed by this enforce form.
   * @param array $known_deps
   *   An array of already-found dependencies, to avoid infinite recursion.
   *
   * @return string[]
   *   An array of all dependencies for $config_name.
   */
  protected function getConfigDependencies(string $config_name, array $known_deps = []): array {
    $all_deps = $this->config($config_name)->get('dependencies');
    if (empty($all_deps)) return [];
    $config_deps = array_key_exists('config', $all_deps) ? $all_deps['config'] : [];
    // Compile a list of known dependencies, so that we can avoid infinite recursion below.
    $known_deps = array_merge($known_deps, $config_deps);
    foreach ($config_deps as $config_dep) {
      // Recurse to ensure we're finding all dependencies.
      $new_deps = $this->getConfigDependencies($config_dep, $known_deps);
      // Strip out any dependencies that we've already found.
      $new_deps = array_diff($new_deps, $known_deps);
      $config_deps = array_merge($config_deps, $new_deps);
    }
    return $config_deps;
  }

  /**
   * Gets messages related to config enforcement operations.
   *
   * @param array $configs
   *   An array of configs, keyed by 'create' and 'update'. Inside 'create' and
   *   'update' is an array of enforcement settings, keyed by config name.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup[]
   *   A list of messages. An empty array if there are no messages to display.
   */
  protected function getOperationMessages(array $configs): array {
    $messages = [];

    if (!empty($configs['update'])) {
      $updated = implode(', ', array_keys($configs['update']));
      $messages[] = $this->t('Updated the enforcement settings for the following configs: %updated', [
        '%updated' => $updated,
      ]);
    }
    if (!empty($configs['create'])) {
      $created = implode(', ', array_keys($configs['create']));
      $messages[] = $this->t('Enforced the following configs: %created', [
        '%created' => $created,
      ]);
    }

    return $messages;
  }

}
