<?php

declare(strict_types=1);

namespace Drupal\Tests\config_enforce_devel\Functional;

use Drupal\config_enforce_devel\TargetModuleBuilder;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;
use Drupal\user\UserInterface;

/**
 * Test behavior of saving Config URIs to the registry.
 *
 * @group config_enforce_devel
 */
class ConfigUriTest extends BrowserTestBase {

  /**
   * The admin user used in this test.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $adminUser;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'config_enforce', 'config_enforce_devel', 'system',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      'administer site configuration',
    ]);

    // Create a target module inside the test site directory, so it gets cleaned
    // up automatically on tearDown() of this test.
    // @see \Drupal\Tests\BrowserTestBase::cleanupEnvironment().
    (new TargetModuleBuilder())
      ->setName(new TranslatableMarkup('Config Enforce - Config URI Test Module'))
      ->setMachineName('config_enforce_config_uri')
      ->setDescription(new TranslatableMarkup('Used for testing Config URIs.'))
      ->setPath($this->siteDirectory . '/modules/custom')
      ->createModule()
      ->installModule()
      ->registerAsTargetModule();
  }

  /**
   * Test that configuration URIs are saved and updated as expected.
   */
  public function testConfigUris(): void {
    $session = $this->getSession();
    $page = $session->getPage();
    /** @var \Drupal\Tests\WebAssert $assert */
    $assert = $this->assertSession();
    $this->drupalLogin($this->adminUser);

    // First, enforce the config directly via the CED config edit form. We
    // expect no config URI to be saved, because no destination query string is
    // set in this case.
    $this->drupalGet('admin/config/development/config_enforce/edit/system.cron');
    $page->checkField('Enforce config');
    $this->submitForm([], 'Save');
    $this->drupalGet('admin/config/development/config_enforce/enforced_configs');
    $assert->pageTextContains('system.cron');

    // Next, enforce the config by following the link (button) from its config
    // form. We expect the config URI to be overwritten with the correct value,
    // because the button sets a destination query string with the URL to the
    // config form.
    $this->drupalGet('admin/config/system/cron');
    $this->clickLink('Manage settings');
    $this->submitForm([], 'Save');
    $this->drupalGet('admin/config/development/config_enforce/enforced_configs');
    $assert->linkByHrefExists('/admin/config/system/cron');

    // Next, edit the enforcement settings directly via the CED config edit form. We
    // expect the existing config URI to be preserved.
    $this->drupalGet('admin/config/development/config_enforce/edit/system.cron');
    $assert->fieldValueEquals('Directory', 'config/optional');
    $page->fillField('Directory', 'config/install');
    $this->submitForm([], 'Save');
    $assert->fieldValueEquals('Directory', 'config/install');
    $this->drupalGet('admin/config/development/config_enforce/enforced_configs');
    $assert->linkByHrefExists('/admin/config/system/cron');

    // Emulate editing via the Enforced configs list. We expect the existing
    // config URI to be preserved, not overwritten with the Enforced configs
    // URL.
    $options = [
      'query' => [
        'destination' => Url::fromRoute('config_enforce_devel.enforced_configs')->toString(),
      ],
    ];
    $this->drupalGet('admin/config/development/config_enforce/edit/system.cron', $options);
    $this->submitForm([], 'Save');
    $this->drupalGet('admin/config/development/config_enforce/enforced_configs');
    $assert->linkByHrefExists('/admin/config/system/cron');

    // Test that an external URL doesn't override the config URI.
    $options = [
      'query' => [
        'destination' => 'https://www.example.com',
      ],
    ];
    $this->drupalGet('admin/config/development/config_enforce/edit/system.cron', $options);
    $this->submitForm([], 'Save');
    $this->drupalGet('admin/config/development/config_enforce/enforced_configs');
    $assert->linkByHrefExists('/admin/config/system/cron');
  }

}
