<?php

declare(strict_types=1);

namespace Drupal\Tests\config_enforce_devel\Functional;

use Drupal\config_enforce\ConfigEnforcer;
use Drupal\config_enforce_devel\TargetModuleBuilder;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Tests\BrowserTestBase;
use Drupal\user\UserInterface;

/**
 * Enforce config by default test.
 *
 * This tests the submit handler mechanism that enforces/updates configs when
 * submitting config forms.
 *
 * @group config_enforce_devel
 */
class EnforceConfigByDefaultTest extends BrowserTestBase {

  /**
   * The admin user used in this test.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $adminUser;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'config_enforce',
    'config_enforce_devel',
    'system',
    'node',
    'field_ui',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      // This permission is added so that the front page can be set to
      // /user/login.
      'link to any page',
      'administer site configuration',
      'administer content types',
      'administer node fields',
      'administer node form display',
    ]);

    // Create a target module inside the test site directory, so it gets cleaned
    // up automatically on tearDown() of this test.
    // @see \Drupal\Tests\BrowserTestBase::cleanupEnvironment().
    (new TargetModuleBuilder())
      ->setName(new TranslatableMarkup('Config Enforce - Enforce by Default Target Module'))
      ->setMachineName('config_enforce_by_default')
      ->setDescription(new TranslatableMarkup('Used for testing Enforce by Default functionality.'))
      ->setPath($this->siteDirectory . '/modules/custom')
      ->createModule()
      ->installModule()
      ->registerAsTargetModule();
  }

  /**
   * Test behaviour when not enforcing by default.
   */
  public function testNoEnforceByDefault(): void {
    $session = $this->getSession();
    $page = $session->getPage();
    /** @var \Drupal\Tests\WebAssert $assert */
    $assert = $this->assertSession();

    // Disable enforce by default.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/development/config_enforce/settings');
    $page->uncheckField('Enforce all configs by default');
    $this->submitForm([], 'Save configuration');

    $this->drupalGet('admin/config/system/site-information');
    $this->submitForm([], 'Save configuration');

    $this->drupalGet('admin/config/development/config_enforce/enforced_configs');
    $assert->pageTextNotContains('system.site');
  }

  /**
   * Test that simple configuration can be enforced by default.
   */
  public function testSimpleEnforceConfigByDefault(): void {
    $session = $this->getSession();
    $page = $session->getPage();
    /** @var \Drupal\Tests\WebAssert $assert */
    $assert = $this->assertSession();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/system/site-information');
    $this->submitForm([], 'Save configuration');
    $assert->statusMessageContains('The configuration options have been saved.', 'status');
    $this->drupalGet('admin/config/development/config_enforce/edit/system.site');
    $this->assertFalse($page->hasField('Enforce config (not yet enforced)'));

    // Disable enforcement for cron config, to test ignore behaviour.
    $this->drupalGet('admin/config/development/config_enforce/edit/system.cron');
    $page->uncheckField('Enforce config (not yet enforced)');
    $this->submitForm([], 'Save');

    // Go save the cron config form, since it's ignored it shouldn't be added to
    // the registry, and should exist in the ignored configs list.
    $this->drupalGet('admin/config/system/cron');
    $this->submitForm([], 'Save');
    $this->drupalGet('admin/config/development/config_enforce/settings');
    $assert->optionExists('edit-ignored-configs', 'system.cron');

    $this->drupalGet('admin/config/development/config_enforce/enforced_configs');
    $assert->pageTextContains('system.site');
    $assert->pageTextNotContains('system.cron');
    $page->checkField('edit-enforced-configs-systemsite');
    $page->pressButton('edit-delete-enforced-configs-submit');
    $assert->statusMessageContains('Deleted enforcement settings for:', 'status');
    $assert->statusMessageContains('system.site', 'status');
    $assert->statusMessageNotContains('system.cron', 'status');

    $this->drupalGet('admin/config/development/config_enforce/enforced_configs');
    $assert->pageTextNotContains('system.site');
    $assert->pageTextNotContains('system.cron');

    // Go back and enforce cron config after previously ignoring it.
    $this->drupalGet('admin/config/development/config_enforce/edit/system.cron');
    $page->checkField('Enforce config (currently ignored)');
    $this->submitForm([], 'Save');
    $this->assertFalse($page->hasField('Enforce config (not yet enforced)'));

    $this->drupalGet('admin/config/system/cron');
    $this->submitForm([], 'Save configuration');
    $assert->statusMessageContains('The configuration options have been saved.', 'status');
    $this->drupalGet('admin/config/development/config_enforce/settings');
    $assert->optionNotExists('edit-ignored-configs', 'system.cron');

    $this->drupalGet('admin/config/development/config_enforce/enforced_configs');
    $assert->pageTextContains('system.cron');
  }

  /**
   * Test that creating a content type and field can be enforced by default.
   */
  public function testContentTypeEnforceConfigByDefault(): void {
    /** @var \Drupal\Tests\WebAssert $assert */
    $assert = $this->assertSession();
    $type = 'enforce_content_type';
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('admin/structure/types/add');
    $edit = [
      'name' => 'Enforce Content Type',
      'type' => $type,
    ];
    $this->drupalGet('admin/structure/types/add');
    $this->submitForm($edit, 'Save');

    // Ensure the content type is enforced initially, not just as a dependency
    // of the field.
    $this->drupalGet("admin/structure/types/manage/{$type}");
    $this->drupalGet('admin/config/development/config_enforce/enforced_configs');
    $assert->pageTextContains("node.type.{$type}");

    // Add an email field.
    $field = 'test_field';
    $this->drupalGet("admin/structure/types/manage/{$type}/fields/add-field");
    $this->clickLink('Email');
    $this->submitForm([], 'Continue');
    $this->submitForm([
      'label' => 'Test field',
      'field_name' => $field,
    ], 'Continue');
    $this->submitForm([], 'Save');

    $this->drupalGet('admin/config/development/config_enforce/enforced_configs');
    $assert->pageTextContains("node.type.{$type}");
    $assert->pageTextContains("field.field.node.{$type}.field_{$field}");
    $assert->pageTextContains("field.storage.node.field_{$field}");
  }

  /**
   * Test that enforcing form display also enforces the content type.
   */
  public function testDependenciesEnforceConfigByDefault(): void {
    /** @var \Drupal\Tests\WebAssert $assert */
    $assert = $this->assertSession();
    $type = 'enforce_dependencies';
    // Create content type programatically, so we can check that it gets pulled
    // in as a dependency.
    $this->createContentType(['type' => $type]);

    // Save form display settings.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet("admin/structure/types/manage/{$type}/form-display");
    $this->submitForm([], 'Save');

    // Check that the content type is also brought in as a dependency.
    $this->drupalGet('admin/config/development/config_enforce/enforced_configs');
    $assert->pageTextContains("node.type.{$type}");
    $assert->pageTextContains("core.entity_form_display.node.{$type}.default");
  }

  /**
   * Test that global settings work as expected with enforce by default.
   */
  public function testNoDependenciesEnforceConfigByDefault(): void {
    $session = $this->getSession();
    $page = $session->getPage();
    /** @var \Drupal\Tests\WebAssert $assert */
    $assert = $this->assertSession();
    $type = 'enforce_no_dependencies';
    // Create content type programmatically.
    $this->createContentType(['type' => $type]);

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/development/config_enforce/settings');
    $page->uncheckField('Enforce dependencies');
    $this->submitForm([], 'Save configuration');

    // Save form display settings.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet("admin/structure/types/manage/{$type}/form-display");
    $this->submitForm([], 'Save');

    // Check that the content type is not brought in as a dependency.
    $this->drupalGet('admin/config/development/config_enforce/enforced_configs');
    $assert->pageTextNotContains("node.type.{$type}");
    $assert->pageTextContains("core.entity_form_display.node.{$type}.default");
  }

  /**
   * Test that global settings work as expected with enforce by default.
   */
  public function testGlobalSettingsEnforceConfigByDefault(): void {
    $session = $this->getSession();
    $page = $session->getPage();
    /** @var \Drupal\Tests\WebAssert $assert */
    $assert = $this->assertSession();
    $type = 'enforce_global_settings';
    // Create content type programmatically.
    $this->createContentType(['type' => $type]);
    $this->drupalGet('admin/config/development/config_enforce/enforced_configs');
    $assert->pageTextNotContains("node.type.{$type}");

    $directory = 'config/install';
    $enforcementLevel = (string) ConfigEnforcer::CONFIG_ENFORCE_NOSUBMIT;

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/development/config_enforce/settings');
    $page->selectFieldOption('Directory', $directory);
    $page->selectFieldOption('Enforcement level', $enforcementLevel);
    $this->submitForm([], 'Save configuration');

    // Save form display settings.
    $this->drupalGet("admin/structure/types/manage/{$type}/form-display");
    $this->submitForm([], 'Save');

    // Check that both configs are enforced with the correct global settings.
    $urls = [
      "admin/config/development/config_enforce/edit/node.type.{$type}",
      "admin/config/development/config_enforce/edit/core.entity_form_display.node.{$type}.default",
    ];
    foreach ($urls as $url) {
      $this->drupalGet($url);
      $assert->fieldValueEquals('Directory', $directory);
      $assert->fieldValueEquals('Enforcement level', $enforcementLevel);
    }
  }

}
