<?php

namespace Drupal\config_layers\Commands;

use Consolidation\OutputFormatters\StructuredData\RowsOfFields;
use Consolidation\OutputFormatters\StructuredData\UnstructuredListData;
use Drupal\config\StorageReplaceDataWrapper;
use Drupal\config_layers\Config\NormalizedStorage;
use Drupal\config_layers\Config\NormalizedStorageComparer;
use Drupal\config_layers\ConfigLayerManager;
use Drupal\config_layers\Entity\ConfigLayer;
use Drupal\Core\Config\ConfigManagerInterface;
use Drupal\Core\Config\FileStorage;
use Drupal\Core\Config\ImportStorageTransformer;
use Drupal\Core\Config\MemoryStorage;
use Drupal\Core\Config\StorageComparer;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drush\Attributes as CLI;
use Drush\Commands\config\ConfigCommands;
use Drush\Commands\config\ConfigImportCommands;
use Drush\Commands\DrushCommands;
use Drush\Drush;
use Drush\Exceptions\UserAbortException;
use Drush\Log\DrushLoggerManager;
use Throwable;

/**
 * Drush commands for the Config Layers module.
 */
class ConfigLayersCommands extends DrushCommands {

  /**
   * Import action.
   */
  const IMPORT = 'import';

  /**
   * Export action.
   */
  const EXPORT = 'export';

  /**
   * The configuration manager.
   *
   * @var \Drupal\Core\Config\ConfigManagerInterface
   */
  protected $configManager;

  /**
   * The active configuration storage.
   *
   * @var \Drupal\Core\Config\StorageInterface
   */
  protected $activeStorage;

  /**
   * The event dispatcher to notify the system that the config was imported.
   *
   * @var \Drupal\config_layers\ConfigLayerManager
   */
  protected $layerManager;

  /**
   * The Drush commands for regular core config imports.
   *
   * @var \Drush\Commands\config\ConfigImportCommands
   */
  protected $configImportCommands;

  /**
   * The import storage transformer.
   *
   * @var \Drupal\Core\config\ImportStorageTransformer
   */
  protected $importStorageTransformer;

  /**
   * The typed configuration manager.
   *
   * @var \Drupal\Core\Config\TypedConfigManagerInterface
   */
  protected $typedConfig;

  /**
   * Constructs a new ConfigSyncCommands object.
   *
   * @param \Drupal\Core\Config\ConfigManagerInterface $config_manager
   *   The configuration manager.
   * @param \Drupal\config_layers\ConfigLayerManager $layerManager
   *   The config layers manager.
   * @param \Drupal\Core\Config\StorageInterface $activeStorage
   *   The active configuration storage.
   * @param \Drupal\Core\Config\ImportStorageTransformer $importStorageTransformer
   *   The import storage transformer.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfig
   *   The typed configuration manager service.
   */
  public function __construct(
    ConfigManagerInterface $config_manager,
    ConfigLayerManager $layerManager,
    StorageInterface $activeStorage,
    ImportStorageTransformer $importStorageTransformer,
    TypedConfigManagerInterface $typedConfig,
  ) {
    parent::__construct();
    $this->configManager = $config_manager;
    $this->layerManager = $layerManager;
    $this->activeStorage = $activeStorage;
    $this->importStorageTransformer = $importStorageTransformer;
    $this->typedConfig = $typedConfig;
    $this->configImportCommands = ConfigImportCommands::create(\Drupal::getContainer());
    $logger = new DrushLoggerManager();
    $this->configImportCommands->setLogger($logger);
  }

  /**
   * Create config layer.
   *
   * @param array $options
   *   Additional options for the command.
   */
  #[CLI\Command(name: 'config-layers:create')]
  #[CLI\Option(name: 'id', description: 'Layer id')]
  #[CLI\Option(name: 'label', description: 'Layer label')]
  #[CLI\Option(name: 'path', description: 'Path to config files.')]
  #[CLI\Option(name: 'reset', description: 'Overwrite configuration instead of merging when enabled.')]
  #[CLI\Option(name: 'weight', description: 'Layer weight.')]
  #[CLI\Option(name: 'source', description: 'Default source to use during import ("active" or "default" = filesystem).')]
  public function layerCreate(
    array $options = [
      'id' => '',
      'label' => '',
      'path' => '',
      'status' => TRUE,
      'reset' => FALSE,
      'weight' => 0,
      'source' => ConfigLayer::CONFIG_SOURCE_DEFAULT,
      'import_mode' => ConfigLayer::UPDATE_MODE_REPLACE,
      'export_mode' => ConfigLayer::UPDATE_MODE_REPLACE,
      'import_events' => [],
      'export_events' => [],
    ],
  ) {
    $layer = ConfigLayer::create($options);
    $layer->save();
  }

  /**
   * Enable layer.
   *
   * @param string $layer_name
   *   The config layer name.
   */
  #[CLI\Command(name: 'config-layers:enable', aliases: ['clen', 'config-layers-enable'])]
  #[CLI\Argument(name: 'layer_name', description: 'The config layer name.')]
  #[CLI\Usage(name: 'drush config-layers:enable active', description: "Enable the 'active' layer.")]
  public function layerEnable($layer_name) {
    $layer = ConfigLayer::load($layer_name);
    if (isset($layer)) {
      $layer->enable();
      $layer->save();
    }
    else {
      $this->logger()->error("Layer named \"$layer_name\" has not been defined.");
    }
  }

  /**
   * Disable layer.
   *
   * @param string $layer_name
   *   The config layer name.
   */
  #[CLI\Command(name: 'config-layers:disable', aliases: ['cldis', 'config-layers-disable'])]
  #[CLI\Argument(name: 'layer_name', description: 'The config layer name.')]
  #[CLI\Usage(name: 'drush config-layers:disable active', description: "Disable the 'active' layer.")]
  public function layerDisable($layer_name) {
    $layer = ConfigLayer::load($layer_name);
    if (isset($layer)) {
      $layer->disable();
      $layer->save();
    }
    else {
      $this->logger()->error("Layer named \"$layer_name\" has not been defined.");
    }
  }

  /**
   * Display a layer config value, or a whole configuration object.
   *
   * @param string $layer_name
   *   The config layer name.
   * @param string $config_name
   *   The config object name, for example "system.site".
   * @param string $key
   *   The config key, for example "page.front". Optional.
   * @param array $options
   *   Additional options for the command.
   */
  #[CLI\Command(name: 'config-layers:get', aliases: ['clget', 'config-layers-get'])]
  #[CLI\Argument(name: 'layer_name', description: 'The config layer name.')]
  #[CLI\Argument(name: 'config_name', description: 'The config object name, for example "system.site".')]
  #[CLI\Argument(name: 'key', description: 'The config key, for example "page.front". Optional.')]
  #[CLI\Option(name: 'source', description: 'The config storage source to read. Additional labels may be defined in settings.php.')]
  #[CLI\Option(name: 'include-overridden', description: 'Apply module and settings.php overrides to values.')]
  #[CLI\Option(name: 'merged', description: 'Get the merged layer.')]
  #[CLI\Usage(name: 'drush config-layers:get active system.site', description: 'Displays the system.site config from the active layer.')]
  #[CLI\Usage(name: 'drush config-layers:get active system.site page.front', description: 'Gets system.site:page.front value from the active layer.')]
  public function get(
    $layer_name,
    $config_name,
    $key = '',
    array $options = [
      'format' => 'yaml',
      'source' => 'active',
      'include-overridden' => FALSE,
      'merged' => FALSE,
    ],
  ) {
    $storage = $this->layerManager->getLayerStorage($layer_name, $options['merged']);
    $config_commands = ConfigCommands::create(Drush::getContainer());
    $config_commands->setConfigFactory($this->layerManager->getConfigFactory($storage));
    $config_commands->setStorage($storage);

    return $config_commands->get($config_name, $key, $options);
  }

  /**
   * Set layer config value directly. Does not perform a config import.
   *
   * @param string $layer_name
   *   The config layer name.
   * @param string $config_name
   *   The config object name, for example "system.site".
   * @param string $key
   *   The config key, for example "page.front".
   * @param string $value
   *   The value to assign to the config key. Use '-' to read from STDIN.
   * @param array $options
   *   Additional options for the command.
   */
  #[CLI\Command(name: 'config-layers:set', aliases: ['clset', 'config-layers-set'])]
  #[CLI\Argument(name: 'layer_name', description: 'The config layer name.')]
  #[CLI\Argument(name: 'config_name', description: 'The config object name, for example "system.site".')]
  #[CLI\Argument(name: 'key', description: 'The config key, for example "page.front".')]
  #[CLI\Argument(name: 'value', description: 'The value to assign to the config key. Use "-" to read from STDIN.')]
  #[CLI\Option(name: 'input-format', description: 'Format to parse the object. Use "string" for string (default), and "yaml" for YAML.')]
  #[CLI\Option(name: 'value', description: 'The value to assign to the config key (if any).')]
  #[CLI\Usage(name: "drush config:set profile system.site page.front '/path/to/page'", description: 'Sets the given path as value for the config item with key "page.front" of "system.site" config object in the "profile" config layer.')]
  public function set(
    $layer_name,
    $config_name,
    $key,
    $value = NULL,
    array $options = [
      'input-format' => 'string',
      'value' => self::REQ,
    ],
  ) {
    $storage = $this->layerManager->getLayerStorage($layer_name);
    $config_commands = ConfigCommands::create(Drush::getContainer());
    $config_commands->setConfigFactory($this->layerManager->getConfigFactory($storage));
    $config_commands->setStorage($storage);

    return $config_commands->set($config_name, $key, $value, $options);
  }

  /**
   * Delete a configuration key, a whole object, or a whole config layer.
   *
   * @param string $layer_name
   *   The config layer name.
   * @param string $config_name
   *   The config object name, for example "system.site" (Optional).
   * @param string $key
   *   A config key to clear, for example "page.front" (Optional).
   */
  #[CLI\Command(name: 'config-layers:delete', aliases: ['cldel', 'config-layers-delete'])]
  #[CLI\Argument(name: 'layer_name', description: 'The config layer name.')]
  #[CLI\Argument(name: 'config_name', description: 'The config object name, for example "system.site". Optional.')]
  #[CLI\Argument(name: 'key', description: 'A config key to clear, for example "page.front". Optional.')]
  #[CLI\Usage(name: 'drush config_layers:delete profile', description: 'Delete the profile layer.')]
  #[CLI\Usage(name: 'drush config_layers:delete profile system.site', description: 'Delete the system.site config object from the profile layer.')]
  #[CLI\Usage(name: "drush config_layers:delete profile system.site page.front", description: "Delete the 'page.front' key from the system.site object in the profile layer.")]
  public function delete($layer_name, $config_name = '', $key = NULL) {
    if (empty($config_name)) {
      $this->layerManager->deleteLayer($layer_name);
    }
    else {
      $storage = $this->layerManager->getLayerStorage($layer_name);
      $config_commands = ConfigCommands::create(Drush::getContainer());
      $config_commands->setConfigFactory($this->layerManager->getConfigFactory($storage));
      $config_commands->setStorage($storage);

      return $config_commands->delete($config_name, $key);
    }
  }

  /**
   * Display status of configuration.
   *
   * @param string $layer_name
   *   The config layer name.
   * @param array $options
   *   Additional options for the command.
   *
   * @return \Consolidation\OutputFormatters\StructuredData\RowsOfFields
   *   Table with configuration status.
   */
  #[CLI\Command(name: 'config-layers:status', aliases: ['clst', 'config-layer-status'])]
  #[CLI\Argument(name: 'layer_name', description: 'The config layer name.')]
  #[CLI\Option(name: 'state', description: 'A comma-separated list of states to filter results.')]
  #[CLI\Option(name: 'prefix', description: 'The config prefix. For example, "system". No prefix returns all names in the system.')]
  #[CLI\Option(name: 'label', description: 'A config directory label (i.e. a key in $config_directories array in settings.php).')]
  #[CLI\Usage(name: 'drush config-layers:status profile', description: 'Display configuration items that need to be synchronized in the profile layer.')]
  #[CLI\Usage(name: 'drush config-layers:status --state=Identical profile', description: 'Display configuration items that are in default state in the profile layer.')]
  #[CLI\Usage(name: "drush config-layers:status --state='Only in sync dir' --prefix=node.type. profile", description: 'Display all content types that would be created in active storage on configuration import in the profile layer.')]
  #[CLI\Usage(name: 'drush config-layers:status --state=Any --format=list profile', description: 'List all config names in the profile layer.')]
  #[CLI\FieldLabels(labels: ['name' => 'Name', 'state' => 'State'])]
  #[CLI\DefaultFields(fields: ['name', 'state'])]
  #[CLI\FilterDefaultField(field: 'name')]
  public function status(
    $layer_name,
    array $options = [
      'state' => 'Only in DB,Only in sync dir,Different',
      'prefix' => self::REQ,
      'label' => self::REQ,
    ],
  ) {
    // Use config layer directory unless label option is specified.
    if (!isset($options['label'])) {
      global $config_directories;
      $label = 'config_layer_' . $layer_name;
      $options['label'] = $label;
      $config_directories[$label] = ConfigLayer::load($layer_name)->getPath();
    }
    $storage = $this->layerManager->getLayerStorage($layer_name);
    $config_commands = ConfigCommands::create(Drush::getContainer());
    $config_commands->setConfigFactory($this->layerManager->getConfigFactory($storage));
    $config_commands->setStorage($storage);

    return $config_commands->status($options);
  }

  /**
   * Display merged configuration information for a set of layers.
   *
   * @param string $layers
   *   Comma-separated list of layer IDs or '*' for all active layers.
   * @param string $config_name
   *   (optional) Configuration object name to filter results.
   * @param array $options
   *   Additional options provided by Drush.
   *
   * @return \Consolidation\OutputFormatters\StructuredData\RowsOfFields|null
   *   Table describing merged configuration and sync status.
   */
  #[CLI\Command(name: 'config-layers:info', aliases: ['clinfo', 'config-layers-info'])]
  #[CLI\Argument(name: 'layers', description: "Comma-separated list of layer IDs or '*' for all active layers.")]
  #[CLI\Argument(name: 'config_name', description: 'Optional configuration object name to filter results.')]
  #[CLI\Option(name: 'show-violations', description: 'Show schema violations.')]
  #[CLI\FieldLabels(labels: ['name' => 'Name', 'layers' => 'Layers', 'synced' => 'Synced', 'schema' => 'Schema'])]
  #[CLI\DefaultFields(fields: ['name', 'layers', 'synced', 'schema'])]
  #[CLI\FilterDefaultField(field: 'name')]
  #[CLI\Usage(name: "drush config-layers:info '*'", description: 'List merged configuration for every active layer.')]
  #[CLI\Usage(name: 'drush config-layers:info gu_profile,portal core.entity_form_display.media.document.default', description: 'Inspect how specific layers impact a single config object.')]
  public function info($layers = '*', $config_name = '', array $options = []): ?RowsOfFields {
    $selection = $this->getLayerEntities($layers);
    $layer_entities = $selection['entities'];
    $target_layers = $selection['targets'];

    $format = $options['format'] ?? 'table';
    $show_violations = $options['show-violations'];

    if (!$layer_entities) {
      $this->logger()->error(dt('No active configuration layers found.'));
      return $format === 'table' ? NULL : new RowsOfFields([]);
    }

    if (!$target_layers) {
      $this->logger()->error(dt('No valid configuration layers matched the provided selection.'));
      return $format === 'table' ? NULL : new RowsOfFields([]);
    }

    $filter = trim((string) $config_name);
    $filter = $filter === '' ? NULL : $filter;

    $active_storage = new NormalizedStorage($this->activeStorage);

    $config_meta = [];
    $merged_values = [];

    foreach ($layer_entities as $entity) {
      $layer_id = $entity->id();
      $layer_storage = new NormalizedStorage($entity->getDatabaseStorage());

      foreach ($this->listConfigObjects($layer_storage) as $config_object) {
        [$collection, $name] = $config_object;
        $key = $this->buildConfigKey($collection, $name);

        $config_data = $this->readFromStorage($layer_storage, $collection, $name);
        if (!is_array($config_data)) {
          continue;
        }

        if (!isset($config_meta[$key])) {
          $config_meta[$key] = [
            'name' => $name,
            'collection' => $collection,
            'layers' => [],
          ];
        }
        if (!in_array($layer_id, $config_meta[$key]['layers'], TRUE)) {
          $config_meta[$key]['layers'][] = $layer_id;
        }

        $should_reset = (bool) $entity->getReset();
        if (!array_key_exists($key, $merged_values) || $should_reset) {
          $merged_values[$key] = $config_data;
        }
        else {
          $merged_values[$key] = ConfigLayerManager::mergeDeepArray($merged_values[$key], $config_data);
        }
      }
    }

    if (!$config_meta) {
      $this->logger()->notice(dt('No configuration found for the provided layers.'));
      return $format === 'table' ? NULL : new RowsOfFields([]);
    }

    $rows = [];
    foreach ($config_meta as $key => $metadata) {
      $has_target_layer = FALSE;
      foreach ($metadata['layers'] as $layer_in_chain) {
        if (isset($target_layers[$layer_in_chain])) {
          $has_target_layer = TRUE;
          break;
        }
      }

      if (!$has_target_layer) {
        continue;
      }

      $display_name = $this->formatConfigName($metadata['collection'], $metadata['name']);

      if (isset($filter) && $filter !== $metadata['name'] && $filter !== $display_name && $filter !== $key) {
        continue;
      }

      $active_target = $metadata['collection'] === StorageInterface::DEFAULT_COLLECTION
        ? $active_storage
        : $active_storage->createCollection($metadata['collection']);

      $active_exists = $active_target->exists($metadata['name']);
      if (!$active_exists) {
        $synced = '-';
      }
      else {
        $active_value = $this->readFromStorage($active_target, StorageInterface::DEFAULT_COLLECTION, $metadata['name']);
        $merged_value = $merged_values[$key] ?? NULL;
        $synced = ($merged_value !== NULL && $active_value == $merged_value) ? 'synced' : 'out-of-sync';
      }

      $rows[$key] = [
        'name' => $display_name,
        'layers' => implode(' → ', $metadata['layers']),
        'synced' => $synced,
        'schema' => $this->summarizeSchemaStatus($metadata['name'], $merged_values[$key] ?? NULL, $show_violations),
      ];
    }

    if (!$rows) {
      $message = isset($filter)
        ? dt('No configuration matches the supplied filter.')
        : dt('The selected layers do not define any configuration.');
      $this->logger()->notice($message);
      return $format === 'table' ? NULL : new RowsOfFields([]);
    }

    ksort($rows);

    return new RowsOfFields($rows);
  }

  /**
   * Import configuration into layer.
   *
   * @param string $layers
   *   (Optional) Comma-separated list of layers to import configuration into
   *   (from filesystem to database). If omitted, import all active layers.
   * @param array $options
   *   Additional options for the command.
   */
  #[CLI\Command(name: 'config-layers:import', aliases: ['clim', 'config-layers-import'])]
  #[CLI\Argument(name: 'layers', description: 'Comma-separated list of layers to import configuration into (optional).')]
  #[CLI\Option(name: 'source', description: 'An arbitrary directory that holds the configuration files.')]
  #[CLI\Option(name: 'from', description: 'active = import from active configuration; default = import from disk (default).')]
  #[CLI\Option(name: 'tag', description: 'Import only layers from the specified tag(s) (comma separated list).')]
  #[CLI\Option(name: 'synchronize', description: 'Synchronize into active configuration.')]
  #[CLI\Option(name: 'update-mode', description: 'replace = delete target configuration and replace with source; copy = keep target configuration, replace only the objects from source; merge = merge source configuration on top of target; merge-target = merge target on top of source, write to target; deduplicate = deduplicate target from duplicated configuration in source; deduplicate-source = deduplicate source from target, write to target.')]
  #[CLI\Usage(name: 'drush config-layers-import', description: 'Import configuration into a configuration layer.')]
  public function layersImport(
    $layers = '',
    array $options = [
      'source' => '',
      'from' => '',
      'synchronize' => FALSE,
      'update-mode' => NULL,
      'tag' => '',
    ],
  ) {

    if (empty($layers)) {
      $tags = empty($options['tag']) ? [] : explode(',', $options['tag']);
      $layers = $this->layerManager->getLayers($tags);
    }
    else {
      $layers = explode(',', $layers);
    }

    foreach ($layers as $layer_name) {
      $layer = ConfigLayer::load($layer_name);
      if (!isset($layer)) {
        $this->logger()->error("Layer named \"$layer_name\" has not been defined.");
      }

      $target_storage = $layer->getDatabaseStorage();

      if (!empty($options['from'])) {
        $layer->setSource($options['from']);
      }

      if (!empty($options['source'])) {
        $layer->setPath($options['source']);
      }

      // N: test against merged layer if importing from active configuration.
      $test_storage = $layer->getSource() == ConfigLayer::CONFIG_SOURCE_ACTIVE ?
        $this->layerManager->getMergedLayer($layer_name) : $target_storage;

      $source_storage = $layer->getSourceStorage();
      $update_mode = $options['update-mode'] ?? $layer->getImportMode();

      if ($this->confirmMessage($source_storage, $test_storage, self::IMPORT, $layer_name)) {
        $this->layerManager->combine($source_storage, $target_storage, $update_mode);
        $this->logger()->success(dt('The configuration was imported successfully into \'!layer\' configuration layer.', ['!layer' => $layer_name]));
      }
    }

    // Merge layers.
    $merged_storage = $this->layerManager->getMergedLayer();

    // Synchronize into active storage.
    if ($options['synchronize']) {
      $merged_storage = $this->importStorageTransformer->transform(new NormalizedStorage($merged_storage));
      $this->synchronizeActive($merged_storage, $this->activeStorage, self::IMPORT, TRUE);
    }
  }

  /**
   * Export config layer(s) to config files.
   *
   * @param string $layers
   *   Comma-separated list of layers to export.
   * @param array $options
   *   Additional options for the command.
   */
  #[CLI\Command(name: 'config-layers:export', aliases: ['clex', 'config-layers-export'])]
  #[CLI\Argument(name: 'layers', description: 'Comma-separated list of layers to export (optional).')]
  #[CLI\Option(name: 'destination', description: 'An arbitrary directory that should receive the exported files.')]
  #[CLI\Option(name: 'tag', description: 'Export only layers from the specified tag(s) (comma separated list).')]
  #[CLI\Option(name: 'merge', description: 'Export the merged result of all layers.')]
  #[CLI\Option(name: 'active-diff', description: 'Export only the changes in active configuration.')]
  #[CLI\Option(name: 'update-mode', description: 'replace = delete target configuration and replace with source; copy = keep target configuration, replace only the objects from source; merge = merge source configuration on top of target; merge-target = merge target on top of source, write to target; deduplicate = deduplicate target from source; deduplicate-source = deduplicate source from target, write to target.')]
  #[CLI\Usage(name: 'drush config-layers-export', description: 'Export configuration layers to disk.')]
  public function layersExport(
    $layers = '',
    array $options = [
      'destination' => NULL,
      'merge' => FALSE,
      'active-diff' => FALSE,
      'update-mode' => NULL,
      'tag' => '',
    ],
  ) {

    if (empty($layers)) {
      $tags = empty($options['tag']) ? [] : explode(',', $options['tag']);
      $layers = $this->layerManager->getLayers($tags);
    }
    else {
      $layers = explode(',', $layers);
    }

    $source_is_layer = TRUE;
    if ($options['merge']) {
      $source_storage = $this->layerManager->getMergedLayer();
      $source_is_layer = FALSE;
    }
    elseif ($options['active-diff']) {
      $source_storage = $this->layerManager->getConfigDiff($this->activeStorage);
      $source_is_layer = FALSE;
    }

    foreach ($layers as $layer_name) {
      $entity = ConfigLayer::load($layer_name);
      if (!isset($entity)) {
        $this->logger()->error("Layer named \"$layer_name\" does not exist.");
      }

      $export_path = $options['destination'] ?? $entity->getPath();
      $file_storage = new FileStorage($export_path);

      $update_mode = $options['update-mode'] ?? $entity->getExportMode();

      if ($source_is_layer) {
        $source_storage = $entity->getDatabaseStorage();
      }

      if ($this->confirmMessage($source_storage, $file_storage, self::EXPORT, $layer_name)) {
        $this->layerManager->combine($source_storage, $file_storage, $update_mode);
        $this->logger()->success(dt('Configuration successfully exported to !target (update-mode=!mode).', [
          '!target' => realpath($export_path),
          '!mode' => $update_mode,
        ]));
      }

      if (!$source_is_layer && $options['destination']) {
        // Only export config once when 'destination' is provided.
        return;
      }
    }
  }

  /**
   * Synchronize layers into active configuration.
   *
   * @param string $layers
   *   Comma-separated list of layers to synchronize,
   *   i.e. partial synchronization of configuration objects (optional).
   */
  #[CLI\Command(name: 'config-layers:synchronize', aliases: ['clsync', 'config-layers-synchronize'])]
  #[CLI\Argument(name: 'layers', description: 'Comma-separated list of layers to synchronize (optional).')]
  public function synchronizeLayers($layers = '') {
    $merged_storage = new NormalizedStorage($this->layerManager->getMergedLayer());

    if (!empty($layers)) {
      // We use a similar strategy to "drush config:import --partial".
      $layers = explode(',', $layers);
      $replacement_storage = new StorageReplaceDataWrapper($this->activeStorage);
      foreach ($layers as $layer_name) {
        $entity = ConfigLayer::load($layer_name);
        $source_storage = $entity->getDatabaseStorage();
        foreach ($source_storage->listAll() as $name) {
          $data = $merged_storage->read($name);
          $replacement_storage->replaceData($name, $data);
        }
      }
      $merged_storage = $replacement_storage;
    }
    else {
      // N: ConfigImportCommands::import() only invokes the import
      // transformer on full imports.
      $merged_storage = $this->importStorageTransformer->transform($merged_storage);
    }

    $this->synchronizeActive($merged_storage);
  }

  /**
   * Reverse synchronize active configuration into layers.
   *
   * Each configuration object in the active storage will be
   * written to the layer where it first occurs (ignoring overlay layers).
   *
   * @param string $layers
   *   Comma-separated list of layers to update (optional).
   * @param array $options
   *   Additional options for the command.
   */
  #[CLI\Command(name: 'config-layers:reverse-synchronize', aliases: ['clrevsync', 'config-layers-reverse-synchronize'])]
  #[CLI\Argument(name: 'layers', description: 'Comma-separated list of layers to update (optional).')]
  #[CLI\Option(name: 'tag', description: 'Export only layers from the specified tag(s) (comma separated list).')]
  public function reverseSynchronizeLayers(
    $layers = '',
    array $options = [
      'tag' => '',
    ],
  ) {
    if (empty($layers)) {
      $tags = empty($options['tag']) ? [] : explode(',', $options['tag']);
      $layers = $this->layerManager->getLayers($tags);
    }
    else {
      $layers = array_filter(array_map('trim', explode(',', $layers)));
    }

    if (empty($layers)) {
      return;
    }

    $tmp_storage = new MemoryStorage();
    ConfigCommands::copyConfig($this->activeStorage, $tmp_storage);
    $source_storage = new NormalizedStorage($tmp_storage);

    $layer_data = [];
    foreach ($layers as $layer_name) {
      $entity = ConfigLayer::load($layer_name);
      if (!$entity) {
        continue;
      }

      $layer_data[] = [
        'weight' => (int) $entity->getWeight(),
        'storage' => new NormalizedStorage($entity->getDatabaseStorage()),
      ];
    }

    if (empty($layer_data)) {
      return;
    }

    usort($layer_data, static function ($a, $b) {
      return $a['weight'] <=> $b['weight'];
    });

    $descending_layers = array_reverse($layer_data);

    foreach ($descending_layers as $layer_info) {
      /** @var \Drupal\config_layers\Config\NormalizedStorage $target_storage */
      $target_storage = $layer_info['storage'];

      $target_collections = ['' => $target_storage];
      foreach ($target_storage->getAllCollectionNames() as $collection_name) {
        $target_collections[$collection_name] = $target_storage->createCollection($collection_name);
      }

      foreach ($target_collections as $collection_name => $target_collection) {
        $source_collection = $collection_name === '' ? $source_storage : $source_storage->createCollection($collection_name);

        foreach ($target_collection->listAll() as $config_key) {
          if (!$source_collection->exists($config_key)) {
            continue;
          }

          $source_config = $source_collection->read($config_key);
          $target_config = $target_collection->read($config_key);

          if (is_array($source_config) && is_array($target_config)) {
            $deduped = ConfigLayerManager::nestedArrayDedupeDeep($source_config, $target_config);
            if (!empty($deduped)) {
              $source_collection->write($config_key, $deduped);
            }
            else {
              $source_collection->delete($config_key);
            }
          }
          elseif ($source_config === $target_config) {
            $source_collection->delete($config_key);
          }
        }
      }
    }

    $source_collections = ['' => $source_storage];
    foreach ($source_storage->getAllCollectionNames() as $collection_name) {
      $source_collections[$collection_name] = $source_storage->createCollection($collection_name);
    }

    foreach ($source_collections as $collection_name => $source_collection) {
      foreach ($source_collection->listAll() as $config_key) {
        $config_data = $source_collection->read($config_key);

        $merged = FALSE;
        foreach ($layer_data as $layer_info) {
          $target_collection = $collection_name === '' ? $layer_info['storage'] : $layer_info['storage']->createCollection($collection_name);
          if (!$target_collection->exists($config_key)) {
            continue;
          }

          $existing = $target_collection->read($config_key);
          $merged_config = ConfigLayerManager::mergeDeep($existing, $config_data);
          $target_collection->write($config_key, $merged_config);
          $merged = TRUE;
          break;
        }

        if (!$merged) {
          $lowest_layer = $layer_data[0];
          $target_collection = $collection_name === '' ? $lowest_layer['storage'] : $lowest_layer['storage']->createCollection($collection_name);
          $existing = $target_collection->exists($config_key) ? $target_collection->read($config_key) : [];
          $merged_config = ConfigLayerManager::mergeDeep($existing, $config_data);
          $target_collection->write($config_key, $merged_config);
        }

        $source_collection->delete($config_key);
      }
    }

  }


  /**
   * Confirmation message. Shows differences in configuration.
   *
   * @param \Drupal\Core\Config\StorageInterface $source_storage
   *   Source storage.
   * @param \Drupal\Core\Config\StorageInterface $target_storage
   *   Target storage.
   * @param string $action
   *   One of self::IMPORT or self::EXPORT.
   * @param string $layer_name
   *   Name of layer.
   */
  protected function confirmMessage(StorageInterface $source_storage, StorageInterface $target_storage, $action, $layer_name) {
    $storage_comparer = new NormalizedStorageComparer($source_storage, $target_storage, $this->configManager);
    if (!$storage_comparer->createChangelist()->hasChanges()) {
      $messages = [
        self::IMPORT => dt('There are no changes to import.'),
        self::EXPORT => dt('There are no changes to export.'),
      ];
      $this->logger()->notice(sprintf('[%s] %s', $layer_name, $messages[$action]));
      return FALSE;
    }

    $change_list = [];
    foreach ($storage_comparer->getAllCollectionNames() as $collection) {
      $change_list[$collection] = $storage_comparer->getChangelist(NULL, $collection);
    }
    $table = ConfigCommands::configChangesTable($change_list, $this->output());
    $table->render();

    $messages = [
      self::IMPORT => dt('Import the listed configuration changes?'),
      self::EXPORT => dt('Export the listed configuration changes?'),
    ];
    if ($this->io()->confirm(sprintf('[%s] %s', $layer_name, $messages[$action]))) {
      return TRUE;
    }
    else {
      throw new UserAbortException();
    }
  }

  /**
   * Synchronize storages.
   *
   * @param \Drupal\Core\Config\StorageInterface $source_storage
   *   Storage to import into the active configuration.
   */
  protected function synchronizeActive(StorageInterface $source_storage) {

    if ($this->confirmMessage($source_storage, $this->activeStorage, self::IMPORT, 'active')) {
      $target_storage = new NormalizedStorage($this->activeStorage);
      $storage_comparer = new StorageComparer($source_storage, $target_storage, $this->configManager);

      $storage_comparer->createChangelist();
      drush_op([$this->configImportCommands, 'doImport'], $storage_comparer);
    }
  }

  /**
   * Resolve layer selection into ordered entities and requested layer set.
   *
   * @param string $layers
   *   Comma-separated list of layer IDs or '*' for all layers.
   *
   * @return array
   *   An array with:
   *   - entities: ordered ConfigLayer entities to inspect.
   *   - targets: associative array of selected layer IDs.
   */
  protected function getLayerEntities(string $layers): array {
    $ordered_ids = array_values($this->layerManager->getLayers());
    $ordered_entities = [];
    $index_map = [];

    foreach ($ordered_ids as $index => $layer_id) {
      $entity = ConfigLayer::load($layer_id);
      if (!isset($entity)) {
        continue;
      }
      $ordered_entities[] = $entity;
      $index_map[$layer_id] = count($ordered_entities) - 1;
    }

    if (!$ordered_entities) {
      return ['entities' => [], 'targets' => []];
    }

    $layers = trim($layers);
    if ($layers === '' || $layers === '*') {
      $targets = array_fill_keys(array_keys($index_map), TRUE);
      return ['entities' => $ordered_entities, 'targets' => $targets];
    }

    $parts = array_values(array_filter(array_map('trim', explode(',', $layers)), 'strlen'));
    $targets = [];
    $max_index = -1;

    foreach ($parts as $layer_id) {
      if (!isset($index_map[$layer_id])) {
        $this->logger()->error(dt('Layer named "@layer" has not been defined or is inactive.', ['@layer' => $layer_id]));
        continue;
      }
      $targets[$layer_id] = TRUE;
      $max_index = max($max_index, $index_map[$layer_id]);
    }

    if ($max_index === -1) {
      return ['entities' => $ordered_entities, 'targets' => []];
    }

    return ['entities' => $ordered_entities, 'targets' => $targets];
  }

  /**
   * Build a unique key for a configuration object.
   */
  protected function buildConfigKey(string $collection, string $name): string {
    return $collection === StorageInterface::DEFAULT_COLLECTION ? $name : $collection . ':' . $name;
  }

  /**
   * Format configuration name for display.
   */
  protected function formatConfigName(string $collection, string $name): string {
    return $collection === StorageInterface::DEFAULT_COLLECTION ? $name : $collection . ':' . $name;
  }

  /**
   * List configuration objects stored within a storage (default + collections).
   */
  protected function listConfigObjects(StorageInterface $storage): array {
    $objects = [];
    foreach ($storage->listAll() as $name) {
      $objects[] = [StorageInterface::DEFAULT_COLLECTION, $name];
    }

    foreach ($storage->getAllCollectionNames() as $collection) {
      $collection_storage = $storage->createCollection($collection);
      foreach ($collection_storage->listAll() as $name) {
        $objects[] = [$collection, $name];
      }
    }

    return $objects;
  }

  /**
   * Summarize the schema validation status for a configuration object.
   */
  protected function summarizeSchemaStatus(string $name, ?array $data, bool $show_violations): string {
    if (!is_array($data)) {
      return '-';
    }

    if (!$this->typedConfig->hasConfigSchema($name)) {
      return 'no schema';
    }

    try {
      $typed = $this->typedConfig->createFromNameAndData($name, $data);
      $violations = $typed->validate();
      $count = count($violations);
      $result = $count === 0 ? 'valid' : sprintf('invalid (%d)', $count);
      if ($show_violations && $count > 0) {
        $result .= PHP_EOL . (string)$violations;
      }
      return $result;
    }
    catch (Throwable $exception) {
      $this->logger()->debug('Schema validation failed for @name: @message', [
        '@name' => $name,
        '@message' => $exception->getMessage(),
      ]);
      return 'error';
    }
  }

  /**
   * Read configuration from storage, returning NULL when absent.
   */
  protected function readFromStorage(StorageInterface $storage, string $collection, string $name): ?array {
    $target = $collection === StorageInterface::DEFAULT_COLLECTION ? $storage : $storage->createCollection($collection);
    $data = $target->read($name);
    return $data === FALSE ? NULL : $data;
  }

}
