<?php

namespace Drupal\config_layers\EventSubscriber;

use Drupal\Component\EventDispatcher\Event;
use Drupal\config_layers\ConfigLayerManager;
use Drupal\config_layers\Entity\ConfigLayer;
use Drupal\Core\Config\ConfigCrudEvent;
use Drupal\Core\Config\ConfigEvents;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Config\StorageTransformEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Config subscriber to import and export layers automatically.
 */
class ConfigSubscriber implements EventSubscriberInterface {

  /**
   * The Drupal Kernel.
   *
   * @var \Drupal\Core\ConfigLayerManager
   */
  protected $layerManager;

  /**
   * ConfigSubscriber constructor.
   *
   * @param \Drupal\config_layers\ConfigLayerManager $layerManager
   *   The layer manager.
   */
  public function __construct(ConfigLayerManager $layerManager) {
    $this->layerManager = $layerManager;
  }

  /**
   * Perform config update action according to event.
   *
   * @param \Drupal\Component\EventDispatcher\Event $event
   *   The event that we want to respond to.
   * @param string $eventName
   *   The event name.
   * @param \Drupal\Core\Config\StorageInterface $source
   *   Source storage to read from.
   * @param \Drupal\Core\Config\StorageInterface $target
   *   Target storage to write to.
   */
  public function updateConfig(Event $event, $eventName, StorageInterface $source, StorageInterface $target) {

    switch ($eventName) {
      case ConfigEvents::SAVE:
        /** @var \Drupal\Core\Config\ConfigCrudEvent $event */
        $config = $event->getConfig();
        $target->write($config->getName(), $config->get());
        break;

      case ConfigEvents::DELETE:
        /** @var \Drupal\Core\Config\ConfigCrudEvent $event */
        $target->delete($event->getConfig()->getName());
        break;

      case ConfigEvents::RENAME:
        /** @var \Drupal\Core\Config\ConfigRenameEvent $event */
        $oldname = $event->getOldName();
        $newname = $event->getConfig()->getName();
        $target->rename($oldname, $newname);
        break;

      default:
        // N: we use this with ConfigEvents::IMPORT.
        $this->layerManager->combine($source, $target, ConfigLayer::UPDATE_MODE_REPLACE);
    }
  }

  /**
   * Import new configuration into layers listening to events.
   *
   * @param \Drupal\Component\EventDispatcher\Event $event
   *   The event to process.
   * @param string $eventName
   *   Name of the event.
   */
  public function configLayerEvents(Event $event, $eventName) {

    // We do not permit updating configuration during config sync.
    if (\Drupal::isConfigSyncing()) {
      return;
    }

    $layer_ids = $this->layerManager->getLayers();
    $layers = ConfigLayer::loadMultiple($layer_ids);

    $updated = FALSE;
    foreach ($layers as $layer) {
      if (in_array($eventName, $layer->getImportEvents())) {
        $this->updateConfig($event, $eventName, $layer->getSourceStorage(), $layer->getDatabaseStorage());
        $updated = TRUE;
      }
    }

    if ($updated) {
      $keys = NULL;
      if ($event instanceof ConfigCrudEvent) {
        $keys = [$event->getConfig()->getName()];
      }
      $this->layerManager->mergeLayers($layers, $keys);
    }

    foreach ($layers as $layer) {
      if (in_array($eventName, $layer->getExportEvents())) {
        if ($event instanceof ConfigCrudEvent) {
          $config_name = $event->getConfig()->getName();
          $source = $layer->getDatabaseStorage();
          $target = $layer->getFileStorage();
          $config = $source->read($config_name);
          if (is_array($config)) {
            $target->write($config_name, $config);
          }
          else {
            $target->delete($config_name);
          }
        }
        else {
          $layer->export();
        }
      }
    }
  }

  /**
   * React to the import transformation.
   *
   * @param \Drupal\Core\Config\StorageTransformEvent $event
   *   The transformation event.
   * @param string $eventName
   *   Name of the event.
   */
  public function configImportTransform(StorageTransformEvent $event, $eventName) {
    $storage = $event->getStorage();
    if ($storage->exists('core.extension')) {
      $core_extension = $storage->read('core.extension');
      $core_extension['module'] = module_config_sort($core_extension['module']);
      $storage->write('core.extension', $core_extension);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    $events[ConfigEvents::SAVE][] = ['configLayerEvents'];
    $events[ConfigEvents::DELETE][] = ['configLayerEvents'];
    $events[ConfigEvents::RENAME][] = ['configLayerEvents'];
    $events[ConfigEvents::IMPORT][] = ['configLayerEvents'];
    $events[ConfigEvents::STORAGE_TRANSFORM_IMPORT][] = ['configImportTransform', 0];
    return $events;
  }

}
