<?php

namespace Drupal\config_pages_overrides\Config;

use Drupal\Component\Utility\NestedArray;
use Drupal\config_pages\ConfigPagesLoaderServiceInterface;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ConfigFactoryOverrideInterface;
use Drupal\Core\Config\Schema\Element;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;

/**
 * Configuration overrides for the profile.
 *
 * @package Drupal\stanford_profile_config_overrides\Config
 */
class ConfigOverrides implements ConfigFactoryOverrideInterface {

  /**
   * Keyed array of config page id to it's configured settings.
   *
   * @var array
   */
  protected $overrideSettings = [];

  /**
   * ConfigOverrides constructor.
   *
   * @param \Drupal\config_pages\ConfigPagesLoaderServiceInterface $configPagesLoader
   *   Config Pages service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config factory service.
   */
  public function __construct(protected ConfigPagesLoaderServiceInterface $configPagesLoader, protected ConfigFactoryInterface $configFactory) {}

  /**
   * {@inheritDoc}
   */
  public function loadOverrides($names) {
    $overrides = [];

    foreach ($this->getOverrideSettings() as $config_page => $config_page_settings) {
      foreach ($config_page_settings as $settings) {
        if (in_array($settings['config_name'], $names)) {
          $path = [$settings['config_name']];
          $path = array_merge($path, explode('.', $settings['config_item']));

          // If overriding a single value grab and use the delta.
          if ($settings['delta'] == FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED) {
            // If overriding an array/map grab all the values.
            $override_value = $this->configPagesLoader->getValue($config_page, $settings['field'], [], $settings['column']);
            $override_value = is_array($override_value) ? array_filter($override_value) : $override_value;
          }
          else {
            // Get the single item.
            $override_value = $this->configPagesLoader->getValue($config_page, $settings['field'], $settings['delta'], $settings['column']);
          }

          // Handle data type stuff.
          if (!is_null($override_value)) {
            if (!empty($settings['prefix']) || !empty($settings['suffix']) && is_string($override_value)) {
              $override_value = $settings['prefix'] . $override_value . $settings['suffix'];
            }

            $override_value = $this->castOverrideValue($override_value, $settings['config_name'], $settings['config_item']);
            NestedArray::setValue($overrides, $path, $override_value);
          }
        }
      }
    }
    return $overrides;
  }

  protected function getOverrideSettings() {
    if ($this->overrideSettings) {
      return $this->overrideSettings;
    }

    $config_page_names = $this->configFactory->listAll('config_pages.type.');
    foreach ($config_page_names as $name) {
      $config_page_type = $this->configFactory->getEditable($name);
      $overrides = $config_page_type->getOriginal('third_party_settings.config_pages_overrides');
      if ($overrides) {
        $this->overrideSettings[$config_page_type->getOriginal('id')] = $overrides;
      }
    }
    return $this->overrideSettings;
  }

  /**
   * Convert the override value based on the schema definition.
   *
   * @param mixed $value
   *   Override value.
   * @param string $config_name
   *   Configuration name.
   * @param $config_item
   *   Path of configuration item.
   *
   * @return mixed
   *   Cast value.
   */
  protected function castOverrideValue($value, $config_name, $config_item) {
    try {
      $config_schema = self::getConfigSchema($config_name);
      $schema = $this->convertConfigElementToList($config_schema);
    }
    catch (\Throwable $e) {
      return $value;
    }
    if (!isset($schema[$config_item])) {
      return $value;
    }

    switch ($schema[$config_item]->getDataDefinition()['type']) {
      case 'boolean':
        return (bool) $value;

      case 'integer':
        return (int) $value ?: strlen((string) $value);
    }

    return $value;
  }

  /**
   * Get the established schema for the current config.
   *
   * @param string $config_name
   *   Config name.
   *
   * @return mixed
   *   The stored schema.
   */
  protected static function getConfigSchema($config_name) {
    return \Drupal::service('config.typed')->get($config_name);
  }

  /**
   * Gets all contained typed data properties as plain array.
   *
   * @param array|object $schema
   *   An array of config elements with key.
   *
   * @return array
   *   List of Element objects indexed by full name (keys with dot notation).
   */
  protected function convertConfigElementToList($schema) {
    $list = [];
    foreach ($schema as $key => $element) {
      if ($element instanceof Element) {
        $list[$key] = $element;
        foreach ($this->convertConfigElementToList($element) as $sub_key => $value) {
          $list[$key . '.' . $sub_key] = $value;
        }
      }
      else {
        $list[$key] = $element;
      }
    }
    return $list;
  }

  /**
   * {@inheritDoc}
   */
  public function createConfigObject($name, $collection = StorageInterface::DEFAULT_COLLECTION) {
    return NULL;
  }

  /**
   * {@inheritDoc}
   */
  public function getCacheableMetadata($name) {
    return new CacheableMetadata();
  }

  /**
   * {@inheritDoc}
   */
  public function getCacheSuffix() {
    return 'ConfigPagesOverrides';
  }

}
