<?php

declare(strict_types=1);

namespace Drupal\config_pages_overrides\Hook;

use Drupal\Core\Cache\Cache;
use Drupal\config_pages\ConfigPagesInterface;
use Drupal\config_pages_overrides\Form\ConfigPagesOverridesAddForm;
use Drupal\config_pages_overrides\Form\ConfigPagesOverridesForm;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;

class ConfigPagesOverridesHooks {

  /**
   * Hook Constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Config Factory service.
   */
  public function __construct(protected EntityTypeManagerInterface $entityTypeManager, protected ConfigFactoryInterface $configFactory) {}

  /**
   * Implements hook_entity_type_alter().
   */
  #[Hook('entity_type_alter')]
  public function entityTypeAlter(array &$entity_types) {
    /** @var \Drupal\Core\Entity\EntityTypeInterface $entity_type */
    $entity_type = $entity_types['config_pages_type'];
    $entity_type->setFormClass('config-override-form', ConfigPagesOverridesForm::class);
    $entity_type->setFormClass('config-override-add-form', ConfigPagesOverridesAddForm::class);
  }

  /**
   * Implements hook_ENTITY_TYPE_presave().
   */
  #[Hook('config_pages_presave')]
  #[Hook('config_pages_delete')]
  public function configPagesCrud(ConfigPagesInterface $page) {
    /** @var \Drupal\config_pages\ConfigPagesTypeInterface $page_type */
    $page_type = $this->entityTypeManager->getStorage('config_pages_type')
      ->load($page->bundle());
    if ($overrides = $page_type->getThirdPartySettings('config_pages_overrides')) {
      $cache_tags = ['route_match'];

      foreach ($overrides as $override) {
        $tags = $this->configFactory->getEditable($override['config_name'])
          ->getCacheTags();
        $this->configFactory->reset($override['config_name']);
        foreach ($tags as $tag) {
          $cache_tags[] = $tag;
        }
      }
      Cache::invalidateTags(array_unique($cache_tags));
    }
  }

}
