<?php

declare(strict_types=1);

namespace Drupal\config_route_requirements\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\RouteSubscriberBase;
use Symfony\Component\Routing\RouteCollection;

/**
 * A route subscriber to remove routes that depend on disabled configuration.
 */
class ConfigRouteSubscriber extends RouteSubscriberBase {

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Configuration factory.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  protected function alterRoutes(
    RouteCollection $collection,
  ): void {
    // Check each route for the _config requirement.
    foreach ($collection as $name => $route) {
      if ($route->hasRequirement('_config')) {
        // Get the value of the requirement. Symfony ensures that it will be a
        // string in \Symfony\Component\Routing\Route::sanitizeRequirement().
        $config_requirements = (string) $route->getRequirement('_config');
        // Evaluate the requirements.
        if (!$this->evaluateRouteConfigRequirements($config_requirements)) {
          // Remove the route from the collection if the evaluation returned
          // FALSE.
          $collection->remove($name);
        }
      }
    }
  }

  /**
   * Evaluate the configuration requirements for a route.
   *
   * Loosely based on the module route subscriber in core.
   *
   * @param string $config_requirements
   *   The contents of the '_config' requirement for a route.
   *
   * @return bool
   *   TRUE if the requirements evaluate to TRUE, FALSE otherwise.
   *
   * @see \Drupal\Core\EventSubscriber\ModuleRouteSubscriber::alterRoutes()
   */
  protected function evaluateRouteConfigRequirements(
    string $config_requirements,
  ): bool {
    // Explode the string on the AND condition, '+'.
    $config_requirements_and = $this->explodeString($config_requirements, '+');
    // Initialize an array for tracking if each AND requirement evaluated to
    // TRUE or FALSE.
    $and_enabled = [];
    foreach ($config_requirements_and as $config_requirement_and) {
      // Explode the AND requirement on the OR condition, ','.
      $config_requirements_or = $this->explodeString($config_requirement_and, ',');
      // Initialize an array for tracking if each OR requirement evaluated to
      // TRUE or FALSE.
      $or_enabled = [];
      foreach ($config_requirements_or as $config_requirement_or) {
        // Get the boolean value of each OR requirement.
        $or_enabled[] = $this->getConfigValue($config_requirement_or);
      }
      // The AND requirement evaluates to TRUE if any of the OR requirements
      // evaluated to TRUE.
      $and_enabled[] = in_array(TRUE, $or_enabled);
    }
    // The requirement as a whole evaluates to TRUE if all of the AND
    // requirements evaluated to TRUE.
    return (bool) array_product($and_enabled);
  }

  /**
   * Explodes a string based on a separator.
   *
   * Borrowed from the module route subscriber in core.
   *
   * @param string $string
   *   The string to explode.
   * @param string $separator
   *   The string separator to explode with.
   *
   * @return array
   *   An array of exploded (and trimmed) values.
   *
   * @see \Drupal\Core\EventSubscriber\ModuleRouteSubscriber::explodeString()
   */
  protected function explodeString(
    string $string,
    string $separator,
  ): array {
    return array_filter(array_map('trim', explode($separator, $string)));
  }

  /**
   * Get the value for a given configuration item.
   *
   * This does not supported nested configuration.
   *
   * @param string $config_requirement
   *   The full configuration name, e.g., 'my_module.settings.foo'.
   *
   * @return bool
   *   TRUE if the value casts to TRUE, FALSE otherwise.
   */
  protected function getConfigValue(
    string $config_requirement,
  ): bool {
    // Store configuration objects and configuration values in static arrays to
    // prevent repeatedly getting the same objects and values.
    $config_objects = drupal_static(__METHOD__ . '_objects', []);
    $config_values = drupal_static(__METHOD__ . '_values', []);
    // Only get the value for this requirement if the result is not yet present
    // in the configuration values static array.
    if (!in_array($config_requirement, $config_values)) {
      // Explode the configuration by '.' to separate the configuration object
      // name and configuration key.
      [$config_namespace, $config_name, $config_key] = explode('.', $config_requirement, 3);
      $config_object_name = $config_namespace . '.' . $config_name;
      // Only get the configuration object if it is not yet present in the
      // configuration objects static array.
      if (!in_array($config_object_name, $config_objects)) {
        $config_objects[$config_object_name] = $this->configFactory->get($config_object_name);
      }
      // Get the value, cast it to boolean, and store the result in the
      // configuration values static array.
      $config_values[$config_requirement] = (bool) $config_objects[$config_object_name]->get($config_key);
    }

    return $config_values[$config_requirement];
  }

}
