<?php

namespace Drupal\Tests\config_route_requirements\Functional;

use Drupal\config_route_requirements\EventSubscriber\ConfigRouteSubscriber;
use Drupal\Tests\BrowserTestBase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use Symfony\Component\Routing\Route;

/**
 * Tests the ConfigRouteSubscriber.
 */
#[CoversClass(
  className: ConfigRouteSubscriber::class,
)]
#[Group(
  name: 'config_route_requirements',
)]
class ConfigRouteSubscriberTest extends BrowserTestBase {

  /**
   * Modules to enable.
   *
   * @var string[]
   */
  protected static $modules = [
    'config_route_requirements',
    'config_route_requirements_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests altering routes.
   *
   * @param array $settings
   *   Settings for the module.
   * @param string[] $enabled_routes
   *   Route names which should be enabled.
   * @param string[] $disabled_routes
   *   Route names which should be disabled.
   */
  #[DataProvider(
    methodName: 'providerTestAlterRoutes',
  )]
  public function testAlterRoutes(
    array $settings,
    array $enabled_routes,
    array $disabled_routes,
  ): void {
    $this->config('config_route_requirements_test.settings')
      ->setData($settings)
      ->save();
    /** @var \Drupal\Core\Routing\RouteBuilderInterface $route_builder */
    $route_builder = $this->container->get('router.builder');
    $route_builder->rebuild();
    /** @var \Symfony\Component\Routing\RouterInterface $router */
    $router = $this->container->get('router');
    $collection = $router->getRouteCollection();

    foreach ($enabled_routes as $enabled_route) {
      $this->assertInstanceOf(
        Route::class,
        $collection->get($enabled_route),
      );
    }
    foreach ($disabled_routes as $disabled_route) {
      $this->assertNull(
        $collection->get($disabled_route),
      );
    }
    $this->assertInstanceOf(Route::class, $collection->get('config_route_requirements_test.no_config_requirement'));
  }

  /**
   * Data provider for testing altering routes.
   *
   * @return array
   *   Array of test data.
   */
  public static function providerTestAlterRoutes(): array {
    $data = [];

    $data[] = [
      'settings' => [
        'setting_one' => TRUE,
        'setting_two' => TRUE,
        'nested_settings' => [
          'setting_three' => TRUE,
          'nested_nested_settings' => [
            'setting_four' => TRUE,
          ],
        ],
      ],
      'enabled_routes' => [
        'config_route_requirements_test.no_config_requirement',
        'config_route_requirements_test.simple',
        'config_route_requirements_test.or',
        'config_route_requirements_test.and',
        'config_route_requirements_test.nested',
        'config_route_requirements_test.nested_nested',
      ],
      'disabled_routes' => [],
    ];

    $data[] = [
      'settings' => [
        'setting_one' => TRUE,
        'setting_two' => FALSE,
        'nested_settings' => [
          'setting_three' => TRUE,
          'nested_nested_settings' => [
            'setting_four' => TRUE,
          ],
        ],
      ],
      'enabled_routes' => [
        'config_route_requirements_test.no_config_requirement',
        'config_route_requirements_test.simple',
        'config_route_requirements_test.or',
        'config_route_requirements_test.nested',
        'config_route_requirements_test.nested_nested',
      ],
      'disabled_routes' => [
        'config_route_requirements_test.and',
      ],
    ];

    $data[] = [
      'settings' => [
        'setting_one' => FALSE,
        'setting_two' => TRUE,
        'nested_settings' => [
          'setting_three' => TRUE,
          'nested_nested_settings' => [
            'setting_four' => TRUE,
          ],
        ],
      ],
      'enabled_routes' => [
        'config_route_requirements_test.no_config_requirement',
        'config_route_requirements_test.or',
        'config_route_requirements_test.nested',
        'config_route_requirements_test.nested_nested',
      ],
      'disabled_routes' => [
        'config_route_requirements_test.simple',
        'config_route_requirements_test.and',
      ],
    ];

    $data[] = [
      'settings' => [
        'setting_one' => FALSE,
        'setting_two' => FALSE,
        'nested_settings' => [
          'setting_three' => TRUE,
          'nested_nested_settings' => [
            'setting_four' => TRUE,
          ],
        ],
      ],
      'enabled_routes' => [
        'config_route_requirements_test.no_config_requirement',
        'config_route_requirements_test.nested',
        'config_route_requirements_test.nested_nested',
      ],
      'disabled_routes' => [
        'config_route_requirements_test.simple',
        'config_route_requirements_test.or',
        'config_route_requirements_test.and',
      ],
    ];

    $data[] = [
      'settings' => [
        'setting_one' => FALSE,
        'setting_two' => FALSE,
        'nested_settings' => [
          'setting_three' => FALSE,
          'nested_nested_settings' => [
            'setting_four' => TRUE,
          ],
        ],
      ],
      'enabled_routes' => [
        'config_route_requirements_test.no_config_requirement',
        'config_route_requirements_test.nested_nested',
      ],
      'disabled_routes' => [
        'config_route_requirements_test.simple',
        'config_route_requirements_test.or',
        'config_route_requirements_test.and',
        'config_route_requirements_test.nested',
      ],
    ];

    $data[] = [
      'settings' => [
        'setting_one' => FALSE,
        'setting_two' => FALSE,
        'nested_settings' => [
          'setting_three' => FALSE,
          'nested_nested_settings' => [
            'setting_four' => FALSE,
          ],
        ],
      ],
      'enabled_routes' => [
        'config_route_requirements_test.no_config_requirement',
      ],
      'disabled_routes' => [
        'config_route_requirements_test.simple',
        'config_route_requirements_test.or',
        'config_route_requirements_test.and',
        'config_route_requirements_test.nested',
        'config_route_requirements_test.nested_nested',
      ],
    ];

    return $data;
  }

}
