# Getting started

!!! note

    It may be helpful to review Drupal's documentation on these topics:

    * [Configuration](https://www.drupal.org/docs/develop/creating-modules/defining-and-using-your-own-configuration-in-drupal)
    * [Controllers](https://www.drupal.org/docs/drupal-apis/routing-system/introductory-drupal-routes-and-controllers-example)
    * [Routes](https://www.drupal.org/docs/drupal-apis/routing-system/routing-system-overview)

## Adding a configuration requirement to a route

### Define configuration schema

Define a boolean configuration value in your module's configuration schema:

```yaml title="config/schema/my_module.schema.yml"
my_module.settings:
  type: 'config_object'
  label: 'My module settings'
  mapping:
    enable_my_route:
      type: 'boolean'
      label: 'Enable my route'
```

!!! warning "Non-boolean configuration values"

    It's recommended to use configuration values that are `boolean`
    [type](https://git.drupalcode.org/project/drupal/-/blob/11.x/core/config/schema/core.data_types.schema.yml).
    Technically, any configuration value can be used, but the value will be
    [converted to a boolean](https://www.php.net/manual/en/language.types.boolean.php#language.types.boolean.casting)
    when the system is deciding if a route should be enabled.

### Create a controller class

Add a simple controller:

```php title="src/Controller/MyRouteController.php"
<?php

declare(strict_types=1);

namespace Drupal\my_module\Controller;

use Drupal\Core\Controller\ControllerBase;

/**
 * Controller.
 */
class MyRouteController extends ControllerBase {

  /**
   * Returns a simple page.
   */
  public function getPage(): array {
    return ['#markup' => $this->t('Page contents')];
  }

}
```

### Define a route for the controller

Define a route and add the `_config` key to the `requirements`:

```yaml title="my_module.routing.yml"
my_module.my_route:
  path: '/my-module/my-route'
  defaults:
    _controller: '\Drupal\my_module\Controller\MyRouteController::getPage'
  requirements:
    _permission: 'administer my module'
    _config: 'my_module.settings.enable_my_route'
```

### Set the configuration value

You can use [Drush](https://www.drush.org/) to easily set the configuration
value:

```shell
# Set my_module.settings.enable_my_route to TRUE to enable the route:
drush config:set my_module.settings enable_my_route 1
# Set my_module.settings.enable_my_route to FALSE to disable the route:
drush config:set my_module.settings enable_my_route 0
```

If `my_module.settings.enable_my_route` is `TRUE`, the route will be
enabled. If it is `FALSE`, the route will not be enabled.

## Adding multiple configuration requirements to a route

!!! note

    Using multiple configuration requirements works similarly to using multiple
    permissions in `_permission` or multiple module dependencies in
    `_module_dependencies`, both of which are
    [provided by Drupal core](https://www.drupal.org/docs/drupal-apis/routing-system/structure-of-routes).

You can specify multiple configuration values separated by `+` for AND logic and
`,` for OR logic.

```yaml title="my_module.routing.yml"
my_module.my_route:
  path: '/my-module/my-route'
  defaults:
    _controller: '\Drupal\my_module\Controller\MyRouteController::get'
  requirements:
    _permission: 'administer my module'
    _config: 'my_module.settings.enable_my_route_1+my_module.settings.enable_my_route_2,my_module.settings.enable_my_route_3'
```

AND logic (`+`) takes precedence over OR logic (`,`). In other words, the
configuration requirement:

```text
my_module.settings.enable_my_route_1+my_module.settings.enable_my_route_2,my_module.settings.enable_my_route_3
```

Would be evaluated as:

```text
my_module.settings.enable_my_route_1 AND (my_module.settings.enable_my_route_2 OR my_module.settings.enable_my_route_3)
```

Which could also be written as:

> All of these requirements must be met to enable the route:
>
> 1. `my_module.settings.enable_my_route_1` must evaluate to `TRUE`
> 2. At least one of these must evaluate to `TRUE`:
>     1. `my_module.settings.enable_my_route_2`
>     2. `my_module.settings.enable_my_route_3`

## Nested configuration values

Nested configuration values are supported.

Define a nested configuration value in your module's configuration schema:

```yaml title="config/schema/my_module.schema.yml"
my_module.settings:
  type: 'config_object'
  label: 'My module settings'
  mapping:
    enable_my_route:
      type: 'boolean'
      label: 'Enable my route'
    nested_settings:
      type: 'config_object'
      label: 'Nested settings'
      mapping:
        enabled_my_route_nested:
          type: 'boolean'
          label: 'Enable my route (nested)'
```

Set the nested configuration value as a requirement on a route:

```yaml title="my_module.routing.yml"
my_module.my_route:
  path: '/my-module/my-route'
  defaults:
    _controller: '\Drupal\my_module\Controller\MyRouteController::get'
  requirements:
    _permission: 'administer my module'
    _config: 'my_module.settings.nested_settings.enabled_my_route_nested'
```
