<?php

namespace Drupal\config_track\EventSubscriber;

use Drupal\config_track\Config\ConfigTrackLanguageConfigOverride;
use Drupal\Core\Config\ConfigManagerInterface;
use Drupal\Core\Config\ConfigCrudEvent;
use Drupal\Core\Config\ConfigEvents;
use Drupal\Core\Config\ConfigRenameEvent;
use Drupal\language\Config\LanguageConfigOverrideCrudEvent;
use Drupal\language\Config\LanguageConfigOverrideEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\TerminateEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * ConfigTrack subscriber for configuration CRUD events.
 */
class ConfigTrackSubscriber implements EventSubscriberInterface {

  /**
   * The config manager.
   *
   * @var \Drupal\Core\Config\ConfigManagerInterface
   */
  protected $configManager;

  /**
   * Constructs the ConfigTrackSubscriber object.
   *
   * @param \Drupal\Core\Config\ConfigManagerInterface $config_manager
   *   The config manager services.
   */
  public function __construct(ConfigManagerInterface $config_manager) {
    $this->configManager = $config_manager;
  }

  /**
   * React to configuration ConfigEvent::SAVE events.
   *
   * @param \Drupal\Core\Config\ConfigCrudEvent $event
   *   The event to process.
   *
   * @throws \Exception
   */
  public function onConfigSave(ConfigCrudEvent $event) {
    $config = $event->getConfig();
    $name = $config->getName();
    // We take care only of simple configs, the others are being covered by
    // entity API hooks in order to maintain the save order properly.
    if ($this->configManager->getEntityTypeIdByName($name) !== NULL) {
      return;
    }

    $config_track =& config_track_static();

    // Only register the shutdown function once.
    if (empty($config_track)) {
      drupal_register_shutdown_function('config_track_shutdown');
    }

    $config_track[] = [
      'name' => $name,
      'operation' => $config->isNew() ? 'create' : 'update',
      'collection' => $config->getStorage()->getCollectionName(),
      'data' => $config->get(),
      'original_data' => $config->getOriginal(),
    ];
  }

  /**
   * React to configuration ConfigEvent::DELETE events.
   *
   * @param \Drupal\Core\Config\ConfigCrudEvent $event
   *   The event to process.
   *
   * @throws \Exception
   */
  public function onConfigDelete(ConfigCrudEvent $event) {
    $config = $event->getConfig();
    $name = $config->getName();
    // We take care only of simple configs, the others are being covered by
    // entity API hooks in order to maintain the save order properly.
    if ($this->configManager->getEntityTypeIdByName($name) !== NULL) {
      return;
    }

    $config_track =& config_track_static();

    // Only register the shutdown function once.
    if (empty($config_track)) {
      drupal_register_shutdown_function('config_track_shutdown');
    }

    $config_track[] = [
      'name' => $name,
      'operation' => 'delete',
      'collection' => $config->getStorage()->getCollectionName(),
      'data' => NULL,
    ];
  }

  /**
   * React to configuration ConfigEvent::RENAMVE events.
   *
   * @param \Drupal\Core\Config\ConfigRenameEvent $event
   *   The event object.
   */
  public function onConfigRename(ConfigRenameEvent $event) {
    $config = $event->getConfig();
    $config_track =& config_track_static();

    // Only register the shutdown function once.
    if (empty($config_track)) {
      drupal_register_shutdown_function('config_track_shutdown');
    }

    $config_track[] = [
      'name' => $event->getOldName(),
      'operation' => 'rename',
      'collection' => $config->getStorage()->getCollectionName(),
      'data' => NULL,
      'debug_backtrace' => [
        'new_name' => $config->getName(),
      ],
    ];
  }

  /**
   * React to configuration LanguageConfigOverrideEvents::SAVE_OVERRIDE events.
   *
   * @param \Drupal\language\Config\LanguageConfigOverrideCrudEvent $event
   *   The event to process.
   *
   * @throws \Exception
   */
  public function onLanguageConfigOverrideSave(LanguageConfigOverrideCrudEvent $event) {
    $config = $event->getLanguageConfigOverride();
    $name = $config->getName();
    $config_track =& config_track_static();

    // Only register the shutdown function once.
    if (empty($config_track)) {
      drupal_register_shutdown_function('config_track_shutdown');
    }

    $config_track[] = [
      'name' => $name,
      'operation' => $config->isNew() ? 'create' : 'update',
      'collection' => $config->getStorage()->getCollectionName(),
      'data' => $config->get(),
      'original_data' => ConfigTrackLanguageConfigOverride::getOriginalByConfig($config),
    ];
  }

  /**
   * React to configuration LanguageConfigOverrideEvents::DELETE_OVERRIDE event.
   *
   * @param \Drupal\language\Config\LanguageConfigOverrideCrudEvent $event
   *   The event to process.
   *
   * @throws \Exception
   */
  public function onLanguageConfigOverrideDelete(LanguageConfigOverrideCrudEvent $event) {
    $config = $event->getLanguageConfigOverride();
    $name = $config->getName();

    $config_track =& config_track_static();

    // Only register the shutdown function once.
    if (empty($config_track)) {
      drupal_register_shutdown_function('config_track_shutdown');
    }

    $config_track[] = [
      'name' => $name,
      'operation' => 'delete',
      'collection' => $config->getStorage()->getCollectionName(),
      'data' => NULL,
    ];
  }

  /**
   * Invoked by the TERMINATE kernel event.
   *
   * @param \Symfony\Component\HttpKernel\Event\TerminateEvent $event
   *   The event object.
   */
  public function onKernelTerminate(TerminateEvent $event) {
    config_track_shutdown();
  }

  /**
   * Registers the methods in this class that should be listeners.
   *
   * @return array
   *   An array of event listener definitions.
   */
  public static function getSubscribedEvents(): array {
    $events[ConfigEvents::SAVE][] = ['onConfigSave', 10];
    $events[ConfigEvents::DELETE][] = ['onConfigDelete', 10];
    $events[ConfigEvents::RENAME][] = ['onConfigRename', 10];

    if (class_exists('\Drupal\language\Config\LanguageConfigOverrideEvents')) {
      $events[LanguageConfigOverrideEvents::SAVE_OVERRIDE][] = ['onLanguageConfigOverrideSave', 10];
      $events[LanguageConfigOverrideEvents::DELETE_OVERRIDE][] = ['onLanguageConfigOverrideDelete', 10];
    }

    // Set a high priority so it is executed before services are destructed. Not
    // doing this would, for example, could result into not executing our
    // logging in case some of the other service destructions throw exceptions
    // and interrupt a regular shutdown.
    $events[KernelEvents::TERMINATE][] = ['onKernelTerminate', 10000];

    return $events;
  }

}
