<?php

namespace Drupal\config_translation_sync\Drush\Commands;

use Drupal\config_translation_sync\ConfigTranslationSyncerInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands for Config Translation Sync.
 */
class ConfigTranslationSyncCommands extends DrushCommands {

  /**
   * Constructs the Drush command class.
   *
   * @param \Drupal\config_translation_sync\ConfigTranslationSyncerInterface $syncer
   *   The config translation syncer service.
   */
  public function __construct(
    private readonly ConfigTranslationSyncerInterface $syncer,
  ) {
    parent::__construct();
  }

  /**
   * Creates an instance of the command class.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   *   A new instance of the command class.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config_translation_sync.syncer'),
    );
  }

  /**
   * Re-synchronize configuration translations.
   */
  #[CLI\Command(name: 'config:resync-translations', aliases: ['crst'])]
  #[CLI\Option(
    name: 'config-names',
    description: 'Comma-separated list of config names or patterns to resync'
  )]
  #[CLI\Option(
    name: 'langcode',
    description: 'Restrict sync to a specific language (or multiple, comma-separated)'
  )]
  #[CLI\Option(
    name: 'exclude',
    description: 'Comma-separated list of configs or patterns to exclude'
  )]
  #[CLI\Usage(
    name: 'drush crst --config-names=system.*',
    description: 'Resynchronize all configs matching system.*'
  )]
  #[CLI\Usage(
    name: 'drush crst --config-names=views.view.* --exclude=views.view.frontpage',
    description: 'Resynchronize all views except the frontpage view'
  )]
  #[CLI\Usage(
    name: 'drush crst --config-names=node.type.* --exclude=node.type.article,node.type.page',
    description: 'Resynchronize all node types except article and page'
  )]
  #[CLI\Usage(
    name: 'drush crst --config-names=system.* --langcode=ru,en',
    description: 'Resynchronize system.* configs for Russian and English'
  )]
  public function resyncTranslations(
    array $options = [
      'config-names' => NULL,
      'langcode' => NULL,
      'exclude' => NULL,
    ],
  ): void {
    $include = !empty($options['config-names'])
      ? array_map('trim', explode(',', $options['config-names']))
      : [];

    $exclude = !empty($options['exclude'])
      ? array_map('trim', explode(',', $options['exclude']))
      : [];

    $this->syncer->sync($include, $exclude, $options['langcode']);
  }

}
