<?php

namespace Drupal\config_translation_sync\Services;

use Drupal\config_translation_sync\ConfigTranslationSyncerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for synchronizing configuration translations.
 */
class ConfigTranslationSyncer implements ConfigTranslationSyncerInterface {

  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected LanguageManagerInterface $languageManager,
    protected StorageInterface $activeStorage,
    protected StorageInterface $syncStorage,
    protected LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function sync(array $include_patterns = [], array $exclude_patterns = [], ?string $langcode = NULL): void {
    $settings = $this->configFactory->get('config_translation_sync.settings');

    // Exclude: use explicitly provided patterns if available,
    // otherwise use global settings.
    $excluded = !empty($exclude_patterns)
      ? $exclude_patterns
      : ($settings->get('excluded_configs') ?? []);

    // Filter the list of configuration items.
    $to_sync = $this->filterConfigs($include_patterns, $excluded);

    if (empty($to_sync)) {
      $this->logger->notice('No configs matched the given patterns.');

      return;
    }

    // Languages: use explicitly provided codes if available,
    // otherwise use global settings, or fall back to all available languages.
    if ($langcode) {
      $target_langcodes = array_map('trim', explode(',', $langcode));
    }
    else {
      $enabled_languages = $settings->get('enabled_languages') ?? [];
      $target_langcodes = !empty($enabled_languages)
        ? $enabled_languages
        : array_keys($this->languageManager->getLanguages());
    }

    foreach ($to_sync as $config_name) {
      foreach ($target_langcodes as $target) {
        $this->syncConfig($config_name, $target);
      }
    }
  }

  /**
   * Filters the list of configuration names by include/exclude patterns.
   *
   * @param string[] $include_patterns
   *   Patterns or names to include.
   * @param string[] $exclude_patterns
   *   Patterns or names to exclude.
   *
   * @return string[]
   *   Filtered list of configuration names.
   */
  protected function filterConfigs(array $include_patterns = [], array $exclude_patterns = []): array {
    $all = $this->syncStorage->listAll();

    $to_sync = empty($include_patterns)
      ? $all
      : $this->applyPatterns($include_patterns, $all);

    if (!empty($exclude_patterns)) {
      $excluded = $this->applyPatterns($exclude_patterns, $all);
      $to_sync = array_diff($to_sync, $excluded);
    }

    return array_values(array_unique($to_sync));
  }

  /**
   * Synchronizes translation for a single configuration item.
   *
   * @param string $config_name
   *   The configuration name.
   * @param string $langcode
   *   The language code.
   */
  protected function syncConfig(string $config_name, string $langcode): void {
    $lang_config_name = "language.$langcode.$config_name";

    if ($this->syncStorage->exists($lang_config_name)) {
      $translation_data = $this->syncStorage->read($lang_config_name);
      $current_data = $this->activeStorage->read($lang_config_name);

      if ($translation_data && $translation_data !== $current_data) {
        $this->activeStorage->write($lang_config_name, $translation_data);
        $this->logger->info('Updated translation for @config in @lang.', [
          '@config' => $config_name,
          '@lang' => $langcode,
        ]);
      }
    }
  }

  /**
   * Applies wildcard patterns (* supported) to a list of configuration names.
   *
   * @param string[] $patterns
   *   Patterns to match.
   * @param string[] $all
   *   All available configuration names.
   *
   * @return string[]
   *   Matching configuration names.
   */
  protected function applyPatterns(array $patterns, array $all): array {
    $result = [];

    foreach ($patterns as $pattern) {
      $regex = '/^' . str_replace('\*', '.*', preg_quote($pattern, '/')) . '$/';

      foreach ($all as $name) {
        if (preg_match($regex, $name)) {
          $result[] = $name;
        }
      }
    }

    return $result;
  }

}
