<?php

namespace Drupal\Tests\config_translation_sync\Unit;

use Drupal\config_translation_sync\Services\ConfigTranslationSyncer;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;

/**
 * Unit tests for the ConfigTranslationSyncer service.
 *
 * @coversDefaultClass \Drupal\config_translation_sync\Services\ConfigTranslationSyncer
 *
 * @group config_translation_sync
 */
class ConfigTranslationSyncerTest extends TestCase {

  /**
   * The config factory mock.
   */
  protected ConfigFactoryInterface|MockObject $configFactory;

  /**
   * The language manager mock.
   */
  protected LanguageManagerInterface|MockObject $languageManager;

  /**
   * The logger mock.
   *
   * @var \Psr\Log\LoggerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected LoggerInterface|MockObject $logger;

  /**
   * The active storage mock.
   *
   * @var \Drupal\Core\Config\StorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected StorageInterface|MockObject $activeStorage;

  /**
   * The sync storage mock.
   *
   * @var \Drupal\Core\Config\StorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected StorageInterface|MockObject $syncStorage;

  /**
   * The service under test.
   */
  protected ConfigTranslationSyncer $syncer;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->languageManager = $this->createMock(LanguageManagerInterface::class);
    $this->logger = $this->createMock(LoggerInterface::class);

    $this->activeStorage = $this->createMock(StorageInterface::class);
    $this->syncStorage = $this->createMock(StorageInterface::class);

    $this->syncer = new ConfigTranslationSyncer(
      $this->configFactory,
      $this->languageManager,
      $this->activeStorage,
      $this->syncStorage,
      $this->logger
    );
  }

  /**
   * Provides scenarios for sync().
   *
   * @return array[]
   *   Test cases.
   */
  public static function providerSync(): array {
    return [
      'translation differs' => [
        'config_name' => 'system.site',
        'langcode' => 'ru',
        'sync_data' => ['name' => 'Сайт'],
        'active_data' => ['name' => 'Site'],
        'expected_writes' => 1,
        'excluded' => [],
      ],
      'translation identical' => [
        'config_name' => 'system.site',
        'langcode' => 'ru',
        'sync_data' => ['name' => 'Сайт'],
        'active_data' => ['name' => 'Сайт'],
        'expected_writes' => 0,
        'excluded' => [],
      ],
      'no translation in sync' => [
        'config_name' => 'system.site',
        'langcode' => 'ru',
        'sync_data' => NULL,
        'active_data' => ['name' => 'Сайт'],
        'expected_writes' => 0,
        'excluded' => [],
      ],
      'excluded config' => [
        'config_name' => 'system.site',
        'langcode' => 'ru',
        'sync_data' => ['name' => 'Сайт'],
        'active_data' => ['name' => 'Site'],
        'expected_writes' => 0,
        'excluded' => ['system.site'],
      ],
    ];
  }

  /**
   * Tests synchronization behavior for different scenarios.
   *
   * @dataProvider providerSync
   */
  public function testSync(
    string $config_name,
    string $langcode,
    ?array $sync_data,
    ?array $active_data,
    int $expected_writes,
    array $excluded,
  ): void {
    $lang = $this->createMock(LanguageInterface::class);
    $lang->method('getId')->willReturn($langcode);
    $this->languageManager->method('getLanguages')
      ->willReturn([$langcode => $lang]);

    $config = $this->createMock(Config::class);
    $config->method('get')->willReturnCallback(function ($key) use ($langcode, $excluded) {
      if ($key === 'enabled_languages') {
        return [$langcode];
      }

      if ($key === 'excluded_configs') {
        return $excluded;
      }

      return [];
    });

    $this->configFactory->method('get')->willReturn($config);
    $this->syncStorage->method('listAll')->willReturn([$config_name]);
    $this->syncStorage->method('exists')->willReturn($sync_data !== NULL);
    $this->syncStorage->method('read')->willReturn($sync_data);
    $this->activeStorage->method('read')->willReturn($active_data);
    $this->activeStorage->expects($this->exactly($expected_writes))
      ->method('write');

    $this->syncer->sync([$config_name], [], $langcode);
  }

}
