<?php

namespace Drupal\config_translation_sync\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for Config Translation Sync.
 */
class ConfigTranslationSyncSettingsForm extends ConfigFormBase {

  /**
   * The language manager service.
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->languageManager = $container->get('language_manager');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['config_translation_sync.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'config_translation_sync_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('config_translation_sync.settings');

    $languages = array_map(
      fn($language) => $language->getName(),
      $this->languageManager->getLanguages()
    );

    $form['enabled_languages'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Enabled languages'),
      '#description' => $this->t('These languages will be used for synchronization.'),
      '#options' => $languages,
      '#default_value' => $config->get('enabled_languages') ?? [],
    ];

    $form['excluded_configs'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Excluded configuration names'),
      '#description' => $this->t('Enter one config name or pattern per line to exclude from synchronization.'),
      '#default_value' => implode("\n", $config->get('excluded_configs') ?? []),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $enabled_languages = array_filter($form_state->getValue('enabled_languages'));
    $excluded_raw = trim($form_state->getValue('excluded_configs'));
    $excluded_configs = $excluded_raw !== ''
      ? preg_split('/\r\n|\r|\n/', $excluded_raw)
      : [];

    $this->config('config_translation_sync.settings')
      ->set('enabled_languages', $enabled_languages)
      ->set('excluded_configs', $excluded_configs)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
