<?php

namespace Drupal\config_uuid_deterministic\Config;

use Drupal\Core\Config\CachedStorage;
use Ramsey\Uuid\Uuid;

/**
 * Cached storage that ensures deterministic UUIDs.
 */
class DeterministicCachedStorage extends CachedStorage {

  use FieldTableNameChecker;

  /**
   * The fixed namespace UUID for deterministic generation.
   */
  private const NAMESPACE = '00000000-0000-0000-0000-000000000000';

  /**
   * {@inheritdoc}
   */
  public function read($name) {
    $data = parent::read($name);
    if ($data && !$this->shouldSkipUuidRemapping($name, $data)) {
      // Always set a deterministic UUID, even if one doesn't exist
      $uuid = Uuid::uuid5(self::NAMESPACE, $name)->toString();

      // Reconstruct array with uuid as first element
      $data = ['uuid' => $uuid] + $data;

      $data = ConfigUuidRemapper::remapNested($data, $name);
    }
    return $data;
  }

  /**
   * {@inheritdoc}
   */
  public function write($name, array $data) {
    if (!$this->shouldSkipUuidRemapping($name, $data)) {
      $uuid = Uuid::uuid5(self::NAMESPACE, $name)->toString();

      // Remove existing uuid if present
      unset($data['uuid']);

      // Reconstruct array with uuid as first element
      $data = ['uuid' => $uuid] + $data;

      $data = ConfigUuidRemapper::remapNested($data, $name);
    }
    return parent::write($name, $data);
  }

  /**
   * {@inheritdoc}
   */
  public function readMultiple(array $names) {
    $list = parent::readMultiple($names);
    foreach ($list as $name => &$data) {
      if ($data && !$this->shouldSkipUuidRemapping($name, $data)) {
        // Always set a deterministic UUID, even if one doesn't exist
        $uuid = Uuid::uuid5(self::NAMESPACE, $name)->toString();

        // Reconstruct array with uuid as first element
        $data = ['uuid' => $uuid] + $data;

        $data = ConfigUuidRemapper::remapNested($data, $name);
      }
    }
    return $list;
  }

}