<?php

namespace Drupal\config_uuid_deterministic\Commands;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drush\Commands\DrushCommands;
use Ramsey\Uuid\Uuid;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands for config UUID deterministic module.
 */
class ConfigUuidDeterministicCommands extends DrushCommands {

  /**
   * The fixed namespace UUID for deterministic generation.
   */
  private const NAMESPACE = '00000000-0000-0000-0000-000000000000';

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a ConfigUuidDeterministicCommands object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(Connection $database, ConfigFactoryInterface $config_factory) {
    parent::__construct();
    $this->database = $database;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('config.factory')
    );
  }

  /**
   * Fix UUIDs in the database to be deterministic.
   *
   * @command config:uuid:fix
   * @usage drush config:uuid:fix
   *   Updates all configuration UUIDs in the database to be deterministic.
   */
  public function fixUuids() {
    $this->output()->writeln('Fixing configuration UUIDs to be deterministic...');

    // Get all config from database
    $query = $this->database->select('config', 'c')
      ->fields('c', ['name', 'data']);
    $results = $query->execute();

    $updated = 0;
    $skipped = 0;

    foreach ($results as $record) {
      $data = unserialize($record->data);

      // Skip field storage configs that would use hashed table names
      if ($this->shouldSkipFieldStorage($record->name, $data)) {
        $this->output()->writeln(sprintf('Skipping %s (uses hashed table names)', $record->name));
        $skipped++;
        continue;
      }

      if (isset($data['uuid'])) {
        $old_uuid = $data['uuid'];
        $new_uuid = Uuid::uuid5(self::NAMESPACE, $record->name)->toString();

        if ($old_uuid !== $new_uuid) {
          // Update the UUID
          $data['uuid'] = $new_uuid;

          // Also update nested UUIDs if present
          $data = $this->remapNestedUuids($data, $record->name);

          // Write back to database
          $this->database->update('config')
            ->fields(['data' => serialize($data)])
            ->condition('name', $record->name)
            ->execute();

          $updated++;
          $this->output()->writeln(sprintf('Updated %s: %s -> %s', $record->name, $old_uuid, $new_uuid));
        }
        else {
          $skipped++;
        }
      }
      else {
        $skipped++;
      }
    }

    $this->output()->writeln(sprintf('Updated %d configurations, skipped %d', $updated, $skipped));

    // Clear all caches
    drupal_flush_all_caches();
    $this->output()->writeln('All caches cleared.');
  }

  /**
   * Check if a field storage config would use hashed table names.
   *
   * @param string $configName
   *   The configuration name.
   * @param array $data
   *   The configuration data.
   *
   * @return bool
   *   TRUE if this is a field storage that would use hashed table names.
   */
  protected function shouldSkipFieldStorage(string $configName, array $data): bool {
    // Only check field.storage.* configs
    if (!str_starts_with($configName, 'field.storage.')) {
      return FALSE;
    }

    // Extract entity type and field name from the config
    if (!isset($data['entity_type']) || !isset($data['field_name'])) {
      return FALSE;
    }

    // Use the same constants as Drupal core
    $tableLengthLimit = 48;
    $entityTypeMaxLength = 32;

    $entityType = substr($data['entity_type'], 0, $entityTypeMaxLength);
    $fieldName = $data['field_name'];

    // Check both regular and revision table names
    $regularTableName = $entityType . '__' . $fieldName;
    $revisionTableName = $entityType . '_revision__' . $fieldName;

    // If either table name would exceed the limit, Drupal uses hashing
    return strlen($regularTableName) > $tableLengthLimit ||
      strlen($revisionTableName) > $tableLengthLimit;
  }

  /**
   * Recursively remap nested UUIDs in configuration data.
   *
   * @param array $data
   *   The configuration data.
   * @param string $configName
   *   The configuration name.
   * @param array $pathSegments
   *   Path segments for nested items.
   *
   * @return array
   *   The remapped data.
   */
  protected function remapNestedUuids(array $data, string $configName, array $pathSegments = []): array {
    $out = [];

    foreach ($data as $key => $value) {
      // Detect plugin instances by presence of both 'id' and 'uuid'.
      if (is_array($value) && isset($value['id'], $value['uuid'])) {
        // Build the new path segments: [..., plugin_id].
        $newPath = array_merge($pathSegments, [$value['id']]);
        // Create a stable name: configName|ancestor1|ancestor2|...|pluginId.
        $stableName = $configName . '|' . implode('|', $newPath);
        // Generate deterministic UUIDv5.
        $newUuid = Uuid::uuid5(self::NAMESPACE, $stableName)->toString();
        // Overwrite the inner uuid.
        $value['uuid'] = $newUuid;
        // Recurse into children under the new UUID key.
        $out[$newUuid] = $this->remapNestedUuids($value, $configName, $newPath);
      }
      elseif (is_array($value)) {
        // Non-plugin nested array: include original key in path.
        $newPath = array_merge($pathSegments, [$key]);
        $out[$key] = $this->remapNestedUuids($value, $configName, $newPath);
      }
      else {
        // Scalar or other: leave unchanged.
        $out[$key] = $value;
      }
    }

    return $out;
  }

  /**
   * Show UUID status for all configurations.
   *
   * @command config:uuid:status
   * @usage drush config:uuid:status
   *   Shows which configurations have non-deterministic UUIDs.
   */
  public function uuidStatus() {
    $this->output()->writeln('Checking configuration UUID status...');

    // Get all config from database
    $query = $this->database->select('config', 'c')
      ->fields('c', ['name', 'data']);
    $results = $query->execute();

    $deterministic = 0;
    $non_deterministic = [];

    foreach ($results as $record) {
      $data = unserialize($record->data);

      if (isset($data['uuid'])) {
        $current_uuid = $data['uuid'];
        $expected_uuid = Uuid::uuid5(self::NAMESPACE, $record->name)->toString();

        if ($current_uuid === $expected_uuid) {
          $deterministic++;
        }
        else {
          $non_deterministic[] = [
            'name' => $record->name,
            'current' => $current_uuid,
            'expected' => $expected_uuid,
          ];
        }
      }
    }

    $this->output()->writeln(sprintf('Deterministic UUIDs: %d', $deterministic));

    if (!empty($non_deterministic)) {
      $this->output()->writeln(sprintf('Non-deterministic UUIDs: %d', count($non_deterministic)));
      $this->output()->writeln('');

      $rows = [];
      foreach ($non_deterministic as $item) {
        $rows[] = [
          $item['name'],
          substr($item['current'], 0, 8) . '...',
          substr($item['expected'], 0, 8) . '...',
        ];
      }

      $this->io()->table(['Config Name', 'Current UUID', 'Expected UUID'], $rows);
    }
    else {
      $this->output()->writeln('All configurations have deterministic UUIDs!');
    }
  }

}