<?php

namespace Drupal\config_uuid_deterministic\Config;

use Ramsey\Uuid\Uuid;

/**
 * Provides UUIDv5 remapping for nested plugin instances in config arrays.
 *
 * Moves the recursive logic for rewriting [id, uuid] pairs into one reusable class.
 */
class ConfigUuidRemapper {

  /**
   * A fixed namespace UUID for all deterministic generation.
   *
   * Replace this with your project’s stable namespace UUID.
   */
  private const NAMESPACE = '00000000-0000-0000-0000-000000000000';

  /**
   * Recursively walks a config array, replacing every nested [id, uuid] pair.
   *
   * @param array $data
   *   The config data slice (possibly nested).
   * @param string $configName
   *   The top-level config name, e.g. 'image.style.thumb__50x50_'.
   * @param array $pathSegments
   *   An array of ancestor plugin IDs or keys (empty at root).
   *
   * @return array
   *   The rewritten data with deterministic UUIDv5 for each plugin instance.
   */
  public static function remapNested(array $data, string $configName, array $pathSegments = []): array {
    $out = [];

    foreach ($data as $key => $value) {
      // Detect plugin instances by presence of both 'id' and 'uuid'.
      if (is_array($value) && isset($value['id'], $value['uuid'])) {
        // Build the new path segments: [..., plugin_id].
        $newPath = array_merge($pathSegments, [$value['id']]);
        // Create a stable name: configName|ancestor1|ancestor2|...|pluginId.
        $stableName = $configName . '|' . implode('|', $newPath);
        // Generate deterministic UUIDv5.
        $newUuid = Uuid::uuid5(self::NAMESPACE, $stableName)->toString();
        // Overwrite the inner uuid.
        $value['uuid'] = $newUuid;
        // Recurse into children under the new UUID key.
        $out[$newUuid] = self::remapNested($value, $configName, $newPath);
      }
      elseif (is_array($value)) {
        // Non-plugin nested array: include original key in path.
        $newPath = array_merge($pathSegments, [$key]);
        $out[$key] = self::remapNested($value, $configName, $newPath);
      }
      else {
        // Scalar or other: leave unchanged.
        $out[$key] = $value;
      }
    }

    return $out;
  }

}
